#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "jastrow.h"




scalar Phik(Vector<scalar> r, Vector<scalar> ck, scalar wk, scalar nuk)
{
  Vector<scalar> rdiff(3);
  scalar dist;

  rdiff = r - ck;
  dist = abs(rdiff);
  return (exp(-dist*dist/(wk*wk + nuk * dist)));
}

scalar drPhik(Vector<scalar> r, Vector<scalar> Ck,
	      scalar wk, scalar nuk)
{
  Vector<scalar> rdiff(3);
  scalar dist;
  scalar phik;
  scalar coeff, denom;
  

  rdiff = r - Ck;
  dist = abs(rdiff);
  phik = Phik(r, Ck, wk, nuk);

  denom = wk * wk + nuk * dist;
  coeff = -dist*(2.0 * wk*wk + nuk * dist)/(denom * denom);

  return(phik*coeff);
}


Vector<scalar> gradPhik(Vector<scalar> r, Vector<scalar> Ck, 
                        scalar wk, scalar nuk)
{
  Vector<scalar> rdiff(3);
  scalar dist;
  scalar drphik = drPhik(r, Ck, wk, nuk);

  rdiff = r - Ck;
  dist = abs(rdiff);
  

  return (drphik /dist * rdiff);
}


scalar d2rPhik(Vector<scalar> r, Vector<scalar> Ck, 
                scalar wk, scalar nuk)
{
  scalar phik = Phik(r, Ck, wk, nuk);
  scalar dist = abs(r - Ck);

  scalar result;
  
  scalar temp1 = dist*(2*wk*wk+nuk*dist);
  scalar temp2 = wk*wk + nuk*dist;
  scalar temp3 = temp1/(temp2*temp2);
  scalar temp2sqr = temp2*temp2;
  
  result = phik*(temp3*temp3 - 2.0*((temp2*temp2sqr - nuk*temp1*temp2)/(temp2sqr*temp2sqr)));
  
  return(result);
}

scalar d2rPhikFD(Vector<scalar> r, Vector<scalar> Ck, 
                scalar wk, scalar nuk)
{
  const scalar delta = 1e-4;
  Vector<scalar> unitvec(3);
  scalar d2r=0.0;

  scalar dist = abs(r - Ck);

  unitvec = (1/dist) * (r - Ck);

  Vector<scalar> rplus(3), rminus(3);

  rplus = r + delta * unitvec;
  rminus = r - delta * unitvec;
  
  d2r = Phik(rplus, Ck, wk, nuk) - 2.0 * Phik(r, Ck, wk, nuk) + Phik(rminus, Ck, wk, nuk);
  d2r = d2r / (delta*delta);
  return (d2r);
}



scalar
jastrow::value(Nvector R)
{

  // Okay, let's do the Slater Determinant part first
  // First, we declare matrices to hold the elements.
  scalar updet, downdet;
  Nvector updetmat(NumUpElecs, NumUpElecs);
  Nvector downdetmat(NumDownElecs, NumDownElecs);
  int NumElecs = NumUpElecs + NumDownElecs;

  // Now we fill in the elements.
  int i,j;
  
  for(i=0; i<NumUpElecs; i++)
    for(j=0; j<NumUpElecs; j++)
      updetmat[i][j] = Phik (R[i], Ck[j], wk[j], nuk[j]);

  for(i=0; i<NumDownElecs; i++)
    for(j=0; j<NumDownElecs; j++)
      downdetmat[i][j] = Phik(R[i+NumUpElecs], Ck[j+NumUpElecs], 
                              wk[j+NumUpElecs], nuk[j+NumUpElecs]);


  // Here we actually calculate the determinants.  Note that we
  // have not yet implemented a determinant for greater that
  // 2x2 matrices.
  updet = determinant (updetmat);
  downdet = determinant (downdetmat);

  scalar lambda = echarge*echarge/D;

  scalar a, b, u;
  scalar dist;
  scalar usum = 0.0;
  for (i=0; i<NumUpElecs; i++)
    {
      a = lambda / 8.0;
      b = sqrt(a)/sqrt(Betanu);
      // Up-Up terms
      for (j=i+1; j<NumUpElecs; j++)
	{
	  dist = abs (R[i] - R[j]);	 
	  usum += a * dist/ (1.0 + b*dist);
	}
      // Up-down terms
      a = lambda / 4.0;
      b = sqrt(a)/sqrt(Betanu);
      for (j=NumUpElecs; j<(NumDownElecs+NumUpElecs); j++)
	{
	  dist = abs (R[i] - R[j]);	  
	  usum += a * dist / (1.0 + b*dist);
	}
      // Up-ion terms
      for (j=0; j<NumIons; j++)
	{
	  a = lambda/2.0;
	  b = sqrt(a)/sqrt(Betanu);
	  dist = abs (R[i] - Rions[j]);	  
	  usum -= Zions[j] * a * dist/ (1.0 + b*dist);
	}      
    }
  for (i=NumUpElecs; i<(NumDownElecs+NumUpElecs); i++)
    {
      // down-down terms
      a = lambda / 8.0;
      b = sqrt(a)/sqrt(Betanu);
      for (j=i+1; j<(NumDownElecs+NumUpElecs); j++)
	{
	  dist = abs (R[i] - R[j]);
	  usum += a * dist / (1.0 + b*dist);
	}
      // down-ion terms
      for (j=0; j < NumIons; j++)
	{
	  a = lambda/2.0;
	  b = sqrt(a)/sqrt(Betanu);
	  dist = abs (R[i] - Rions[j]);	  
	  usum -= Zions[j] * a * dist / (1.0 + b*dist);
	}
    }

  return (updet * downdet  * exp(usum) );
}




scalar
jastrow::LocalEnergy(Nvector R)
{

  // DEBUG
  //  printf ("R vector:\n");
  // R.print();
  
  // We need N 3-D vectors to hold the gradients with respect
  // to each of the elctronic coordinates.  We split this up into 
  // a terms from the Uij's and a term from the determinants.
  const int NumElecs = NumUpElecs + NumDownElecs;

  Nvector GradU(NumElecs, 3), 
          GradDet(NumElecs, 3), 
          TotalGrad(NumElecs, 3);
  Vector<scalar> LaplacianU(NumElecs), LaplacianDet(NumElecs);
  
  // First we calculate the gradient of the Uij sum.
  // We do the sum in parts.
  
  scalar aij, bij;
  scalar lambda = echarge*echarge/D;
  
  for (int i=0; i < NumElecs; i++)
    {
      // First, zero out the gradient and Laplacian.
      GradU[i][0] = GradU[i][1] = GradU[i][2] = 0.0;
      LaplacianU[i] = 0.0; 
      // First, do the electron-electron terms
      
      for (int j=0; j < NumElecs; j++)
	{
	  if (i != j)
	    {
	      if (((i<NumUpElecs) && (j<NumUpElecs)) ||
		  ((i>=NumUpElecs) && (j>=NumUpElecs))) // Same spin terms
		aij = lambda / 8.0;
	      else
		aij = lambda / 4.0;
	      
	      bij = sqrt(aij) / sqrt(Betanu);
	      scalar rij = abs(R[i] - R[j]);
	      scalar temp = 1.0 + bij * rij;
	      scalar mag = aij / (rij * temp * temp);
              GradU[i] = GradU[i] + mag * (R[i] - R[j]);
	      LaplacianU[i] += 2.0 * aij / (rij * temp * temp * temp);
	    }
	}
      // Now do the electron-ion terms
      for (int j=0; j<NumIons; j++)
	{
	  aij = lambda/2.0;
	  bij = sqrt(aij)/sqrt(Betanu);
	  
	  scalar rij = abs(R[i] - Rions[j]);
	  scalar temp = 1.0 + bij * rij;
	  scalar mag = - Zions[j] * aij / (rij * temp * temp);
	  GradU[i] = GradU[i] + mag * (R[i] - Rions[j]);
	  LaplacianU[i] -= 2.0 * Zions[j] * aij / (rij * temp * temp * temp);
	}
    }
  

  // Now for the nasty part of the calculation:  the gradient and
  // Laplacian of the determinants.
  // Let's first allocate the up and down Slater matrices:
  Nvector Aup(NumUpElecs, NumUpElecs);
  Nvector Adown(NumDownElecs, NumDownElecs);

  // Let's make an array of matrices to store the 3 components of the 
  // gradient matrix.

  Nvector gradAup[3];
  Nvector gradAdown[3];
  // Now initialze the sizes of the matrices
  gradAup[0].init(NumUpElecs, NumUpElecs);
  gradAup[1].init(NumUpElecs, NumUpElecs);
  gradAup[2].init(NumUpElecs, NumUpElecs);

  gradAdown[0].init(NumDownElecs, NumDownElecs);
  gradAdown[1].init(NumDownElecs, NumDownElecs);
  gradAdown[2].init(NumDownElecs, NumDownElecs);

  // Now let's fill up the A matrices the gradient component matrices.

  for (int k=0; k<NumUpElecs; k++)
    for (int l=0; l<NumUpElecs; l++)
      {
	Vector<scalar> gradAkl(3);
        gradAkl  = gradPhik(R[l], Ck[k], wk[k], nuk[k]);
	gradAup[0][k][l] = gradAkl[0];
	gradAup[1][k][l] = gradAkl[1];
	gradAup[2][k][l] = gradAkl[2];
	
	Aup[k][l] = Phik(R[l], Ck[k], wk[k], nuk[k]);
      }

  for (int k=0; k<NumDownElecs; k++)
    for (int l=0; l<NumDownElecs; l++)
      {
	Vector<scalar> gradAkl(3);
	gradAkl = gradPhik(R[l+NumUpElecs], Ck[k+NumUpElecs], 
                           wk[k+NumUpElecs], nuk[k+NumUpElecs]);
	gradAdown[0][k][l] = gradAkl[0];
	gradAdown[1][k][l] = gradAkl[1];
	gradAdown[2][k][l] = gradAkl[2];

	Adown[k][l] = Phik(R[l+NumUpElecs], Ck[k+NumUpElecs], 
                         wk[k+NumUpElecs], nuk[k+NumUpElecs]);
      }
  // Now let's calculate the inverse of Aup and Adown
  Nvector Aupinv(NumUpElecs, NumUpElecs);
  Nvector Adowninv(NumDownElecs, NumDownElecs);

  Aupinv = inverse (Aup);
  Adowninv = inverse (Adown);

  // Now let's calculate the elements of the gradients.  Each element 
  // will be the dot product of a row in Ainverse by the corresponding 
  // column in gradA.

  for (int i=0; i<NumUpElecs; i++)
    {
      for (int comp=0; comp<3; comp++)
	{
	  scalar sum=0.0;
	  for (int j=0; j<NumUpElecs; j++)
	    sum += Aupinv[i][j] * gradAup[comp][j][i];
	  GradDet[i][comp] = sum;
	}
    }

  for (int i=0; i<NumDownElecs; i++)
    {
      for (int comp=0; comp<3; comp++)
	{
	  scalar sum=0.0;
	  for (int j=0; j<NumDownElecs; j++)
	    sum += Adowninv[i][j] * gradAdown[comp][j][i];
	  GradDet[i+NumUpElecs][comp] = sum;
	}
    }



  // Next, we need to calculate the Laplacian of the determinants.  This
  // will be the nastiest part of the calculation. The Laplacian of the 
  
  // There are two parts to the calculation:
  // \nabla^2_{r_i} \ln[\det(A)] = Tr[A^{-1} \nabla^2^{r_i} A]
  //                              -Tr[(A^{-1} \nabla_{r_i} A)^2]

  // First, we calculate the Laplacian of the elements of A.  We 
  // recall that A_kl = phi_k(r_l).  
  // \nabla^2{r_l} A_kl = 2/rlk d/d rlk phik(rlk) + d^2/drlk^2 phik(rlk)
  
  Nvector LaplacianAup(NumUpElecs, NumUpElecs);
  Nvector LaplacianAdown(NumDownElecs, NumDownElecs);
  
  // First, we fill up the matrix containing the Laplacians of the phik(rl).
  for(int k=0; k < NumUpElecs; k++)    // Loop over up orbitals
    for(int l=0; l < NumUpElecs; l++)  // Loop over up electrons
      {
	Vector<scalar> rlk(3); 
	rlk  = R[l] - Ck[k];
	scalar dist = abs(rlk);
	
	LaplacianAup[k][l] = 2.0 / dist * drPhik(R[l], Ck[k], wk[k], nuk[k])
	  + d2rPhik(R[l], Ck[k], wk[k], nuk[k]);
      }

  for(int k=0; k < NumDownElecs; k++)    // Loop over down orbitals
    for(int l=0; l < NumDownElecs; l++)  // Loop over down electrons
      {
	Vector<scalar> rlk(3);
	rlk = R[l+NumUpElecs] - Ck[k+NumUpElecs];
	scalar dist = abs(rlk);
	
	LaplacianAdown[k][l] = 2.0 / dist * drPhik(R[l+NumUpElecs], 
						 Ck[k+NumUpElecs], 
						 wk[k+NumUpElecs], 
						 nuk[k+NumUpElecs])	
	  + d2rPhik(R[l+NumUpElecs], Ck[k+NumUpElecs], 
		    wk[k+NumUpElecs], nuk[k+NumUpElecs]);
      }
  
  // Now we have everything we need to calculate the Laplacians of the
  // determinants.  The Laplacian with respect to r_m has two terms.
  // The first term is just Tr(A^{-1} \nabla^2_{r_m} A).  We begin
  // with this term.  The laplacian of A w.r.t. r_m has zeros in all
  // but its m^th column.  We have stored all the Laplacians in one
  // matrix for convenience, but we must remember that when we want to
  // calculate the Laplacian with respect to r_m, we must simply dot
  // the m^th row of Ainv with the m^th column of LaplacianA.

  for (int m=0; m < NumUpElecs; m++)     // Loop over up electrons
    {
      LaplacianDet[m] = 0.0;
      for (int k=0; k < NumUpElecs; k++)   // Dummy index for dot product
	LaplacianDet[m] += Aupinv[m][k] * LaplacianAup[k][m];
    }

  for (int m=0; m < NumDownElecs; m++)     // Loop over down electrons
    {
      LaplacianDet[m+NumUpElecs] = 0.0;
      for (int k=0; k < NumDownElecs; k++)   // Dummy index for dot product
	LaplacianDet[m+NumUpElecs] += Adowninv[m][k] * LaplacianAdown[k][m];
    }


  // Now we have to do the second term.  This one, after some slight
  // manipulations, is quite simple.  It is just the square of the
  // gradient of the log of the determinant that we calculated above.
  // We have to subtract this term off the one just calculated.

  for (int m=0; m < NumElecs; m++)
    for (int comp=0; comp < 3; comp++)
      LaplacianDet[m] -= GradDet[m][comp] * GradDet[m][comp];

  // Now we have all of the necessary components to construct the
  // local kinetic energy.  We need merely sum of the Laplacians and
  // the square of the gradients.  We begin by computing the total
  // gradient, which is the sum of the U gradient and the determinant
  // gradient.

  for (int m=0; m < NumElecs; m++)
    TotalGrad[m] =  GradU[m] +  GradDet[m];
  
  scalar KineticE = 0.0;
  
  for (int m=0; m < NumElecs; m++)
    KineticE += -D * (LaplacianU[m] + LaplacianDet[m] +
		     dot_prod(TotalGrad[m], TotalGrad[m]));
	
  //scalar GradSqr = 0.0;
  //for (int m=0; m<NumElecs; m++)
  // GradSqr += dot_prod(TotalGrad[m], TotalGrad[m]);
  
  // printf ("Analytic Grad Squared term = %1.18f.\n", GradSqr);
  
  //scalar Laplacian = 0.0;
  //for (int m=0; m<NumElecs; m++)
  // Laplacian += LaplacianDet[m];

  // printf ("Analytic Laplacian term = %1.18f.\n", Laplacian);


  
  // Now we need to calculate the potential energy
  // We need to loop over all pairs of particles.
  
  scalar PotentialE = 0.0;

  for (int e1=0; e1<NumElecs; e1++)
    {
      // First, the electron-electron terms
      for (int e2=e1+1; e2<NumElecs; e2++)
	{
	  Vector<scalar> rdiff(3);
	  rdiff = R[e1] - R[e2];
	  scalar dist = abs(rdiff);	  
	  PotentialE += echarge * echarge / dist;
	}
      // Now, the electron-ion terms
      for (int z=0; z<NumIons; z++)
	{
	  Vector<scalar> rdiff(3);
	  rdiff = R[e1] - Rions[z];
	  scalar dist = abs(rdiff);	  
	  PotentialE -= echarge * echarge * Zions[z] / dist;
	}
    }
  // Finally, the ion-ion terms
  for (int z1=0; z1<NumIons; z1++)
    for (int z2=z1+1; z2<NumIons; z2++)
      {
	Vector<scalar> rdiff(3);
	rdiff = Rions[z1] - Rions[z2];
	scalar dist = abs(rdiff);	  
	
	PotentialE += echarge * echarge * Zions[z1] * Zions[z2] / dist;
      }
  
  
  return (KineticE + PotentialE);
  
}

// This is a finite difference version of the local energy
// calculation, which we will use to check our analytic version.  If
// this works on the first time, I will eat my desk.
scalar
jastrow::LocalEnergyFD(Nvector R)
{
  scalar Laplacian = 0.0;
  const scalar delta = 1e-4;
  const scalar invsqdelta = 1.0 / (delta*delta);
  int NumElecs = NumUpElecs + NumDownElecs;

  for (int m=0; m<NumElecs; m++)
    for (int comp=0; comp<3; comp++)
      {
	Nvector Rtemp(NumElecs, 3);
	Rtemp = R;
	Rtemp[m][comp] += delta;
	Laplacian += invsqdelta * value(Rtemp);
	Laplacian -= 2.0 * invsqdelta * value(R);
	Rtemp = R;
	Rtemp[m][comp] -= delta;
	Laplacian += invsqdelta * value(Rtemp);
      }

  scalar KineticE = -D * Laplacian / value(R);

  return (KineticE);
}

scalar
jastrow::LocalEnergyFD2(Nvector R)
{
  int NumElecs = NumUpElecs + NumDownElecs;  
  scalar Laplacian = 0.0;
  Nvector Grad(NumElecs,3);
  const scalar delta = 1e-2;
  const scalar invsqdelta = 1.0 / (delta*delta);
  
  scalar GradSqr = 0.0;

  for (int m=0; m<NumElecs; m++)
    for (int comp=0; comp<3; comp++)
      {
	Nvector Rplus(NumElecs, 3), Rminus(NumElecs,3);
	Rplus = Rminus = R;	
	Rplus[m][comp] += delta;
	Rminus[m][comp] -= delta;

	//printf ("  %1.12f \n- %1.12f\n", log(value(Rplus)) - log(value(R)),
	//log(value(R)) - log(value(Rminus)));  

	Grad[m][comp] = (log (value(Rplus)) - log(value(R))) / delta;
	
	Laplacian += invsqdelta * log(value(Rplus));
	Laplacian -= 2.0 * invsqdelta * log(value(R));
	Laplacian += invsqdelta * log(value(Rminus));

	//printf("Laplacian/value = %1.12f\n", Laplacian / invsqdelta  / log(value(R)));
      }
  for (int m=0; m<NumElecs; m++)
    for (int comp=0; comp<3; comp++)
      GradSqr += Grad[m][comp] * Grad[m][comp];

  // printf ("Grad Squared term = %1.18f\n", GradSqr);
  // printf ("Laplacian term = %1.18f\n", Laplacian);

  scalar KineticE = -D * (Laplacian + GradSqr);

  return (KineticE);
}

// jastrow::read(char *filename) -- reads in the data for a moleculte from
// the file specified by filename.  The file should have the following format:

// NumUpElecs NumDownElecs
// NumIons
// Rion1(3) Rion2(3) Rion3(3)...
// Zion1 Zion2 Zion3...
// Ck1(3) Ck2(3) ...
// w1 w2 w3...
// nu1 nu2 nu3...
// Betanu

int 
jastrow::read(char *filename)
{
  FILE *fin;
  
  fprintf (stderr, "Got to read function.\n");
  
  printf ("filename = %s\n", filename);
  
  
  if ((fin = fopen(filename, "r")) == NULL)
    die ("Can't open the molecule file.  Exitting.\n");
  
  fscanf(fin, "%d %d\n", &NumUpElecs, &NumDownElecs);
  int NumElecs = NumUpElecs + NumDownElecs;
   
  fscanf(fin, "%d \n", &NumIons);
  
  Rions.init(NumIons,3);
  Zions.newsize(NumIons);
  Ck.init(NumElecs, 3);
  wk.newsize(NumElecs);
  nuk.newsize(NumElecs);
  
  for (int ion=0; ion < NumIons; ion++)
    for (int comp=0; comp < 3; comp++)
      fscanf (fin, "%lf ", &Rions[ion][comp]);
  fscanf (fin, "\n");
  
  for (int ion=0; ion < NumIons; ion++)
      fscanf (fin, "%lf ", &Zions[ion]);
  
  for (int k=0; k<NumElecs; k++)
    for (int comp=0; comp < 3; comp++)
      fscanf (fin, "%lf ", &Ck[k][comp]);
 
  for (int k=0; k<NumElecs; k++)
    fscanf (fin, "%lf ", &wk[k]);
  for (int k=0; k<NumElecs; k++)
    fscanf (fin, "%lf ", &nuk[k]);

  fscanf(fin, "%lf ", &Betanu);
  
  printf("Molecule file successfully processed!\n");
  printf("Number of Up/Down Electrons = %d / %d\n",NumUpElecs,NumDownElecs);
  printf("Number of Ions = %d\n",NumIons);

  printf("Ion#  \t     X         Y        Z \t   charge  betanu \n");
  for (int i=0; i<NumIons; i++)
    printf("%d \t %8.4f %8.4f %8.4f %8.1f \t %8.4f\n", i, Rions[i][0], Rions[i][1], Rions[i][2], Zions[i], Betanu);

  printf("Electron     wk \t     nuk \t             ck \n");
  for (int i=0; i<NumElecs; i++)
    printf("%d \t %8.4f \t %8.4f \t %8.4f %8.4f %8.4f\n", i, wk[i], nuk[i], Ck[i][0], Ck[i][1], Ck[i][2]);
  

  return(1);
}



scalar
jastrow::NodalLinkAction (Nvector R, Nvector Rprime, scalar tau)
{
  Vector<scalar> unitvec(3);
  scalar nodeaction = 0.0;
  
  if ((NumUpElecs > 2) || (NumDownElecs > 1))
    die ("General Nodal action not implemented.\n");

  for (int k=0; k<NumUpElecs; k++)
    if (fabs(nuk[k]) > 1e-6)
      die ("General Nodal action not implemented.\n");
  
  unitvec = (Ck[0] - Ck[1]) * (1.0 / abs(Ck[0] - Ck[1]));
  
  scalar dist, distprime;
  dist = dot_prod(R[0] - R[1], unitvec);
  distprime = dot_prod(Rprime[0]-Rprime[1], unitvec);

  if ((dist < 0.0) || (distprime < 0.0))
    return (1.0e5);  // Node crossing
  
  // Note the mystery factor of 2.0!!!!
  return (-log(1.0 - exp (-dist*distprime / (2.0 * D * tau))));
}



