#include "md.h"

/*  Force calculation related routines for the MD program. */

double Wall_Perturb(double A,double lam_x,double x,double lam_y,double y,double gam_dot,double t)
{
    double delta;
    /*  Calculates the distance our "rough" wall is perturbed from a flat,
        planar wall. */
    delta=A*(sin(2*PI*(x+gam_dot*t)/lam_x)*sin(2*PI*y/lam_y));
    return(delta);
}

double Wall_Force(double x, double y, double z, double sigma, double eps_w, double m, double n, double A, double lam_x, double lam_y, double gam_dot, double t, double wall)
{
    double z_0, r_perp, r_red, force;

    /*  Find the displacement of the wall's base plane from our function. */
    z_0=Wall_Perturb(A,lam_x,x,lam_y,y,gam_dot,t);
  
    /*  The effective radius is r-displacement. */
    r_perp=fabs(wall/2-fabs(z+z_0));

    if(r_perp<=0)
    {
        printf("Zero radius.\n");
        printf("Radius: %lf.\n",r_perp);
        printf("z, z_0: %lf, %lf.\n",z,z_0);
    }

    
  
    /*  This is sigma/(r-displacement). */
    r_red=sigma/r_perp;
  
    force=(4*eps_w/r_perp)*(m*pow(r_red,m)-n*pow(r_red,n));
    if(fabs(force)>20.0)
    printf("Radius: %lf, Force: %lf\t",r_perp,force);
    return(force);
}

double dFdx(double A, double lam_x, double x, double lam_y, double y, double t, double gam_dot)
{
  
    double deriv;
  
    /*  This subroutine calculates the derivateves of the displacement
        function of the wall's base plane along the x direction, which
        is used to calculate the potential gradient of the wall along
        the x direction. */
  
    /*  The below function is the ANALYTIC partial derivative of the wall
        perturbation function in the x direction.  E.g. if the wall 
        perturbation function was f(x,y)=sin(2*PI*x)+cos(2*PI*y) then the
        resulting function should be:  2*PI*cos(2*PI*x). */
  
    deriv=A*2*PI*sin(2*PI*(x+gam_dot*t)/lam_x)*sin(2*PI*y/lam_y)/lam_x;
    return(deriv);
}

double dFdy(double A, double lam_x, double x, double lam_y, double y, double t, double gam_dot)
{
  
    double deriv;
  
    /*  This subroutine calculates the derivateves of the displacement
        function of the wall's base plane along the y direction, which
        is used to calculate the potential gradient of the wall along
        the y direction. */
  
    /*  The below function is the ANALYTIC partial derivative of the wall
        perturbation function in the y direction.  E.g. if the wall 
        perturbation function was f(x,y)=sin(2*PI*x)+cos(2*PI*y) then the
        resulting function should be:  -2*PI*sin(2*PI*y). */
  
    deriv=-A*2*PI*sin(2*PI*y/lam_y)/lam_y*sin(2*PI*(x+gam_dot*t)/lam_x);
    return(deriv);    
}  

void Calc_Forces(Atom3D_t* p, double rc_wall, double A, double lam_x, double lam_y, double gam_eff, double t, int N, double eps_w, double eps, double wall)
{
    double base_force, radmag, pot, r_red, r_red6, r_red12, cutoff, rc;
    double x, y, z;
    int i,j, parity;
    Vector3D_t dr, F;


    /*  First off, we'll calculate the wall interaction forces of all
        molecules within the cutoff radius (rc_wall) of the wall. */
    for(i=0;i<N;i++)
    {
        /*  First of all, we'll zero out the forces since we calculate
            forces fresh each iteration. */
        p[i].F.z=0.0;
        p[i].F.y=0.0;
        p[i].F.x=0.0;

        x=p[i].R.x;
        y=p[i].R.y;
        z=p[i].R.z;

        /*  We'll choose to keep the bottom wall stationary.  The easiest
            way to do it is to set gam_eff = 0.0. */
        if(z<0)
        gam_eff=0.0;
        
        cutoff=rc_wall+Wall_Perturb(A,lam_x,x,lam_y,y,gam_eff,t);
        /*  If a particle is within the range of the wall's influence,
            we'll calculate the force on it due to the wall. */
        if(fabs(p[i].R.z)>=((wall/2)-cutoff))
        {
            /* Base_pot is dU(x,y,z)/dz.  dFdx is df(x,y)dx,
               dFdy is df(x,y) dy.  f(x,y) is the wall perturbation
               function.  The chain rule is needed for the x, y
               forces. */

	  /*  If the wall is the positive wall, the actual direction of
	      force is in the NEGATIVE direction. */
	  if(p[i].R.z>0)
          parity=-1;
	  else
          parity=1;
          base_force=parity*
          Wall_Force(x,y,z,p[i].sigma,eps_w,9,4,A,lam_x,lam_y,gam_eff,t,wall);
          p[i].F.z=base_force;
          if (fabs(p[i].F.z)>20.0)
          printf("N:%d Z: %lf Force: %lf\n",i,p[i].R.z,p[i].F.z);
          p[i].F.x=base_force*(-dFdx(A,lam_x,x,lam_y,y,t,gam_eff));
          p[i].F.y=base_force*(-dFdy(A,lam_x,x,lam_y,y,t,gam_eff));
      }
      
        /* Now we calculate the interatomic forces working on this atom
           due to the other atoms. */
      
        /* Simple (NON-optimized) atom-atom pair potential calculations. */
        for(j=(i+1);j<N;j++)
        {
            /*  Find the vector from particle i to particle j. */
            /*  The odd math is to find the minimum image, since we 
                actually always store the UNWOUND coordinates in the
                particle records. */
            dr.x=p[j].R.x-p[i].R.x;
            dr.y=p[j].R.y-p[i].R.y;
            dr.z=p[j].R.z-p[i].R.z;
            
            /*  Find the magnitude of the vector (find R for the potential) */
            radmag=sqrt(dr.x*dr.x+dr.y*dr.y+dr.z*dr.z);
            rc=p[j].sigma*2.5;
            if(radmag<=rc)
            {
                r_red=p[j].sigma/radmag;

                /*  Find the potential itself. */
                r_red6=r_red*r_red*r_red*r_red*r_red*r_red;
                r_red12=r_red6*r_red6;
                pot=(4*eps/radmag)*(12*r_red12-6*r_red6);
            
                /*  rad.x/radmag is the directional component of the i->j 
                    interaction.  Thus pot*(rad.x/radmag) is the x force
                    component acting on particle i.  Since F(i)=-F(j), we can
                    calculate two forces per loop. */
                F.x=(dr.x/radmag)*pot;
                F.y=(dr.y/radmag)*pot;
                F.z=(dr.z/radmag)*pot;
                p[i].F.x+=F.x;
                p[j].F.x-=F.x;
                p[i].F.y+=F.y;
                p[j].F.y-=F.y;
                p[i].F.z+=F.z;
                p[j].F.z-=F.z;
            }
            
        }
    }
    return;
}

