#include "md.h"

/*  Initialization functions for the MD simulation. */

int Initialize_Program(int *N, double L, double z, double A, double eta, Atom3D_t** particles, Scalar_Avg_t** v_prof, double** rho, Scalar_Avg_t** S, Vector3D_t*** vmat, double** vac, int maxlag, int maxbin, double layer_size, int* num_layers,Vector3D_t** KE, int tmax)
{
    int rho_bin_size,i;
    double V_cell, V_atom, drho;
    
    /*  First we need to find N, which we can find from the packing 
        fraction requested.  Note that L and z are both entered in terms
        of sigma, so we don't need to explicitly account for sigma here.*/
    V_cell=L*L*z;
    /*  The volume of a single atom is 4/3*PI*R^3, and R is (sigma/2).
        Therefore, the volume of an atom in terms of sigma is PI/6*sigma^3. */
    V_atom=PI/6.;
    /*  Eta=N*V_atom/V_cell; Therefore, N=Eta*V_Cell/V_Atom. */
    (*N)=floor(eta*V_cell/V_atom);
    if((*N)>MAX_N)
    {
        printf("The requested packing fraction or cell dimensions lead ");
        printf("to too many atoms in the cell.\nExiting.\n");
        return(INIT_ERR);
    }

    
    /*  Now that we know how many particles we have, we can allocate their
        memory. */
    if(NULL==((*particles)=(Atom3D_t *)malloc((*N)*sizeof(Atom3D_t))))
    {
        ALLOC_ERROR("particle array");
    }
    
    /*  Now the array to store the velocity profile.  First we determine
        how many "layers" we'll consider the velocity to be in.  Layer_size,
        like all spatial measurements, is in terms of sigma. */
    layer_size=0.25;
    (*num_layers)=(int) (z+2*A)/layer_size;

    /*  Now we'll allocate storage space for the array to store the
        average values. */
    if(NULL==((*v_prof)=(Scalar_Avg_t *)malloc((*num_layers)*sizeof(Scalar_Avg_t))))
    {
        ALLOC_ERROR("velocity profile array");
    }

    /*  And now an array for the average density.  Again, we need to find
     the number of bins.  Let's look at increments of 0.10 sigma to start.
     The longest distance in this case is a half cell radius, L/2. */
    
    drho=0.10;
    rho_bin_size=(int) (L/2)/drho;
    if(NULL==((*rho)=(double *)malloc(rho_bin_size*sizeof(double))))
      {
	ALLOC_ERROR("density profile array");
      }

    /*  We need to allocate memory for S(k) too. */
    if(NULL==((*S)=(Scalar_Avg_t *)malloc(maxbin*sizeof(Scalar_Avg_t))))
    {
        ALLOC_ERROR("S(k) array");
    }

    /*  And the Velocity autocorrelation array.. */
    if(NULL==((*vac)=(double *)malloc(maxlag*sizeof(double))))
    {
        ALLOC_ERROR("velocity autocorrelation array");
    }
    /*  And finally we need to allocate the velocity matrix. */
    if(NULL==((*vmat)=(Vector3D_t **)matrix_alloc((*N),maxlag,sizeof(Vector3D_t *),sizeof(Vector3D_t))))
    {
        ALLOC_ERROR("velocity matrix for Vac");
    }
    
    if(NULL==((*KE)=(Vector3D_t*)malloc(tmax*sizeof(Vector3D_t))))
    {
        ALLOC_ERROR("kinetic energy array");
    }
    for(i=0;i<tmax;i++)
    {
        (*KE)[i].x=(*KE)[i].y=(*KE)[i].z=0.0;
    }
    
    return(NO_ERR);
}

void normalvar(double *n1, double *n2)
{
    double u1, u2;
    /*  Generates two random normal variates from two random uniform
        variates, as discussed in Allen and Tildesley, p. 347. */

    u1=drand48();
    u2=drand48();
    *n1=sqrt(-2*log(u1))*cos(2*PI*u2);
    *n2=sqrt(-2*log(u1))*sin(2*PI*u2);
    return;
}

void* matrix_alloc(int m, int n, size_t row_size, size_t element_size)
{
    void** base;
    int i,j;

    if(NULL==(base=(void *)malloc(m*row_size)))
    {
        printf("Couldn't allocate the initial pointer for the matrix.\n");
        return(base);
    }
    for(i=0;i<n;i++)
    {
        if(NULL==(base[i]=(void *)malloc(n*element_size)))
        {
            for(j=(i-1);j>=0;j--)
            free(base[j]);
            free(base);
        }
    }
    return(base);
}

void Cleanup(Atom3D_t* p, Scalar_Avg_t* vz, double* rho, Scalar_Avg_t* S, Vector3D_t** vmat, double* vac, int N)
{
    int i;

    free(p);
    free(vz);
    free(rho);
    free(S);
    free(vac);
    for(i=0;i<N;i++)
    free(vmat[i]);
    free(vmat);
    return;
}


void Initialize_System(double sigma, double eta, double L, double wall, double A, int N, double mnorm, Atom3D_t* p)
{

    double sigma_s,x_i,y_i,z_i,x,y,z,x_max,y_max,z_max,nx,ny,nz,dummy;
    int parity, i;
    /*  Set up the initial configuration in an FCC lattice.  Sigma_s is 
        a scaled distance between parts, based on a maximal packing
        fraction of 0.74 for a fully dense FCC lattice. */
    sigma_s=pow((eta/0.74),(1./3.))*sigma;

    /*  Parity tells us if we're on an A or B plane in the FCC lattice. */
    parity=0;
    x_i=(-L/2.0+sigma_s)+parity*(sigma_s/2);
    y_i=(-L/2.0+sigma_s);
    z_i=(-wall)/2.0+2*A+sigma_s;
    x_max=y_max=(L-sigma_s)/2.;
    z_max=(wall-sigma_s-A)/2.;
    
    i=0;
    z=z_i;
    while((z<z_max)&&(i<N))
    {
        y=y_i;
        while((y<y_max)&&(i<N))
        {
            x=x_i+parity*(sigma_s/2);
            while((x<x_max)&&(i<N))
            {
                /* Set the initial lattice positions. */
                p[i].R.x=p[i].U.x=x;
                p[i].R.y=p[i].U.y=y;
                p[i].R.z=p[i].U.z=z;
                /* And the initial velocities too. */
                normalvar(&nx,&ny);
                normalvar(&nz,&dummy);
                p[i].v.x=mnorm*nx;
                p[i].v.y=mnorm*ny;
                p[i].v.z=mnorm*nz;
                /* Initialize everything else to zero. */
                p[i].a.x=p[i].b.x=p[i].c.x=p[i].F.x=0.0;
                p[i].a.y=p[i].b.y=p[i].c.y=p[i].F.y=0.0;
                p[i].a.z=p[i].b.z=p[i].c.z=p[i].F.z=0.0;
                x+=sigma_s;
                i++;
            }
            y+=sigma_s;
            parity=(parity++)%2;
        }
        z+=sigma_s;
    }
    
    /*  We probably don't _need_ to fill these in, since we assume
        this is true elsewhere in the simulation, but it's always a
        good idea to do things properly anyway.  Plus, this way if
        we wanted to change the mass of some atoms, we easily could. */
    for(i=0;i<N;i++)
    {
        p[i].sigma=1.0;
        p[i].mass=1.0;
    }
    return;
}









