#include "md.h"

/*  Input/output functions for the molecular dynamics simulation. */

int Read_Input(char* filename, double* temp, double* sigma, double* L, double* z, double* eta, double* gam_dot, double* A, double* lam_x, double* lam_y, double* eps, double* eps_w, double* layer_size, int*rseed, int*tmax, int*datacount)
{
    FILE* infile;

    if(NULL==(infile=fopen(filename,"r")))
    {
        FILE_OPEN_ERR(filename);
    }
    if(0>=fscanf(infile,"Temp: %le\n",temp))
    {
        INPUT_ERROR("temperature");
    }
    if(0>=fscanf(infile,"Sigma: %le\n",sigma))
    {
        INPUT_ERROR("atomic diameter");
    }
    if(0>=fscanf(infile,"Cell Length: %le\n",L))
    {
        INPUT_ERROR("cell size");
    }
    if(0>=fscanf(infile,"Wall Width: %le\n",z))
    {
        INPUT_ERROR("wall width");
    }
    if(0>=fscanf(infile,"Packing Fraction: %le\n",eta))
    {
        INPUT_ERROR("packing fraction");
    }
    if(0>=fscanf(infile,"Shear Rate: %le\n",gam_dot))
    {
        INPUT_ERROR("shear rate");
    }
    if(0>=fscanf(infile,"Max Wall Feature: %le\n",A))
    {
        INPUT_ERROR("wall feature amplitude");
    }
    if(0>=fscanf(infile,"X Wavelength: %le\n",lam_x))
    {
        INPUT_ERROR("wall feature x wavelength");
    }
    if(0>=fscanf(infile,"Y Wavelength: %le\n",lam_y))
    {
        INPUT_ERROR("wall feature y wavelength");
    }
    if(0>=fscanf(infile,"Inter-atomic Epsilon: %le\n",eps))
    {
        INPUT_ERROR("atomic epsilon");
    }
    if(0>=fscanf(infile,"Inter-wall Epsilon: %le\n",eps_w))
    {
        INPUT_ERROR("wall epsilon");
    }
    if(0>=fscanf(infile,"Random Number Seed: %d\n",rseed))
    {
        INPUT_ERROR("random rumber seed");
    }
    if(0>=fscanf(infile,"Velocity Layer Size: %le\n",layer_size))
    {
        INPUT_ERROR("velocity profile layer size");
    }
    if(0>=fscanf(infile,"Maximum Number of Steps: %d\n",tmax))
    {
        INPUT_ERROR("maximum number of steps");
    }
    if(0>=fscanf(infile,"Steps Per Calculation: %d\n",datacount))
    {
        INPUT_ERROR("data calculation rate");
    }
    
    return(NO_ERR);
}

int Write_Energy(Vector3D_t* KE, int tmax,char* filename)
{
    FILE *outfile;
    char *KEname;
    int i;

    KEname=strpaste(filename, "KE");
    if(NULL==(outfile=fopen(KEname,"w")))
    {
        FILE_OPEN_ERR(KEname);
    }
    for(i=0;i<tmax;i++)
    {
        fprintf(outfile,"%d   %le   %le   %le\n",i,KE[i].x,KE[i].y,KE[i].z);
    }
    fflush(outfile);
    fclose(outfile);
    return(NO_ERR);
}

    
int Write_Continue(Atom3D_t* p, int N, char* filename, Vector3D_t** vmat, int maxlag, int t)
{
    FILE *outfile;
    char *contname;
    int i,j;
   
    contname=strpaste(filename, "cont");
    if(NULL==(outfile=fopen(contname,"w")))
    {
        FILE_OPEN_ERR(contname);
    }

    /*  Output each particle's values. */
    for(i=0;i<N;i++)
    {
        fprintf(outfile,"%le\t %le\n",p[i].sigma,p[i].mass);
        fprintf(outfile,"%le\t %le\t %le\n",p[i].U.x,p[i].U.y,p[i].U.z);
        fprintf(outfile,"%le\t %le\t %le\n",p[i].R.x,p[i].R.y,p[i].R.z);
        fprintf(outfile,"%le\t %le\t %le\n",p[i].v.x,p[i].v.y,p[i].v.z);
        fprintf(outfile,"%le\t %le\t %le\n",p[i].a.x,p[i].a.y,p[i].a.z);
        fprintf(outfile,"%le\t %le\t %le\n",p[i].b.x,p[i].b.y,p[i].b.z);
        fprintf(outfile,"%le\t %le\t %le\n",p[i].c.x,p[i].c.y,p[i].c.z);
        fprintf(outfile,"%le\t %le\t %le\n",p[i].F.x,p[i].F.y,p[i].F.z);
    }
    fprintf(outfile,"%d\t%d\n",t,maxlag);
    for(i=0;i<maxlag;i++)
    {
        for(j=0;j<N;j++)
        fprintf(outfile,"%le  %le  %le\t",vmat[j][i].x,vmat[j][i].y,vmat[j][i].z);
        fprintf(outfile,"\n");
    }
    fflush(outfile);
    fclose(outfile);
    return(NO_ERR);
}

int Read_Continue(Atom3D_t *p, int N, char* filename, Vector3D_t** vmat, int* maxlag, int* t)
{
    FILE *infile;
    int i,j;

    if(NULL==(infile=fopen(filename,"r")))
    {
        FILE_OPEN_ERR(filename);
    }

    /*  Read each particle's values. */
    for(i=0;i<N;i++)
    {
        fscanf(infile,"%le\t %le\n",&p[i].sigma, &p[i].mass);
        fscanf(infile,"%le\t %le\t %le\n",&p[i].U.x, &p[i].U.y, &p[i].U.z);
        fscanf(infile,"%le\t %le\t %le\n",&p[i].R.x, &p[i].R.y, &p[i].R.z);
        fscanf(infile,"%le\t %le\t %le\n",&p[i].v.x, &p[i].v.y, &p[i].v.z);
        fscanf(infile,"%le\t %le\t %le\n",&p[i].a.x, &p[i].a.y, &p[i].a.z);
        fscanf(infile,"%le\t %le\t %le\n",&p[i].b.x, &p[i].b.y, &p[i].b.z);
        fscanf(infile,"%le\t %le\t %le\n",&p[i].c.x, &p[i].c.y, &p[i].c.z);
        fscanf(infile,"%le\t %le\t %le\n",&p[i].F.x, &p[i].F.y, &p[i].F.z);
    }    
    fscanf(infile,"%d\t%d\n",t,maxlag);

    /*  Read in the velocity autocorrelation matrix. */
    for(i=0;i<(*maxlag);i++)
    {
        for(j=0;j<N;j++)
        fscanf(infile,"%le  %le  %le\t",&vmat[j][i].x,&vmat[j][i].y,&vmat[j][i].z);
        fscanf(infile,"\n");
    }
    fclose(infile);
    return(NO_ERR);
}


int Write_Output(Scalar_Avg_t* S, double* Vac, Scalar_Avg_t* Vz, int Sbin, int maxlag, int num_layers, double dk, double layer_size, int t, char* filename)
{
    FILE *Sout, *Vacout, *Vzout;
    char *Sname, *Vacname, *Vzname, *timechar;
    int i;

    if(NULL==(timechar=(char *)malloc(10*sizeof(char))))
    {
        ALLOC_ERROR("character string conversion");
    }
    
    itoa(t,timechar);
    Sname=strpaste(filename,"S");
    Sname=strpaste(Sname,timechar);
    Vacname=strpaste(strpaste(filename,"Vac"),timechar);
    Vzname=strpaste(strpaste(filename,"Vz"),timechar);

    /*  Write out the S(k) to a file named:  Filename.S.t. */
    if(NULL==(Sout=fopen(Sname,"w")))
    {
        FILE_OPEN_ERR(Sname);
    }
    for(i=0;i<Sbin;i++)
    {
        fprintf(Sout,"%.10le\t%.10le\n",(dk*i),S[i].val);
    }
    fflush(Sout);
    fclose(Sout);
    
    /*  Write out <V(0).V(t)> to a file named:  Filename.Vac.t. */
    if(NULL==(Vacout=fopen(Vacname,"w")))
    {
        FILE_OPEN_ERR(Vacname);
    }
    for(i=0;i<maxlag;i++)
    {
        fprintf(Vacout,"%d\t%.10le\n",i,Vac[i]);
    }
    fflush(Vacout);
    fclose(Vacout);

    /*  Write out V(z) to a file named:  Filename.Vz.t. */
    if(NULL==(Vzout=fopen(Vzname,"w")))
    {
        FILE_OPEN_ERR(Vzname);
    }
    for(i=0;i<num_layers;i++)
    {
        fprintf(Vzout,"%.10le\t%.10le\n",i*layer_size,Vz[i].val);
    }
    fflush(Vzout);
    fclose(Vzout);
    return(NO_ERR);
}






