// ************************************************************
// *                                                          *
// *  list225.h                                               *
// *                                                          *
// *  This file provides the interface for our abbreviated    *
// *  version of the STL list class. The code is not          *
// *  optimized to the extent that a "real" STL               *
// *  implementation would be, but it has much of the same    *
// *  functionality, and has more understandable source code. *
// *                                                          *
// *  Written July 1999 by Jason Zych                         *
// *                                                          *
// ************************************************************ 

#ifndef LIST_H
#define LIST_H

#include <stddef.h>
#include <iostream.h>


template <class Etype> class list;
template<class Etype> std::ostream& 
      operator<< (std::ostream&, const list<Etype>&);


template <class Etype>
class list
{
public:



   // ************************
   // *                      *
   // *  typedefs            *
   // *                      *
   // ************************

   typedef Etype value_type;
   typedef Etype* pointer;
   typedef const Etype* const_pointer; 
   typedef Etype& reference; 
   typedef const Etype& const_reference;


   // Declarations/definitions for
   //    iterator
   //    const_iterator
   //    reverse_iterator
   //    const_reverse_iterator
   // appear below.
   //
   // The declarations  
   //    allocator_type
   //    size_type
   //    difference_type
   //  will not be used in this class.



 // *** iterator for the list class

protected:

   class listNode;
   typedef listNode* listNodePtr;

public:
   
   // *********************************************
   // *                                           *
   // *  list<>::iterator                         *
   // *                                           *
   // *  iterator for our list class. The code    *
   // *  should ideally be in the .C file, with   *
   // *  only the declarations in this file, but  * 
   // *  the compiler (and indeed, most C++       *
   // *  compilers) do not yet support such an    *
   // *  arrangement for nested classes, so we    *
   // *  have the iterator code in this file.     *
   // *                                           *
   // *  Written July 1999 by Jason Zych          *
   // *                                           *
   // *********************************************

   class iterator 
   {
   public:


      typedef Etype value_type;
      typedef Etype* pointer;
      typedef Etype& reference;

      // iterator
      //    - constructor
      //    - returns an iterator initialized to nothing
      iterator() { ptr = NULL;}


      // operator*
      //    - return type : a reference to the element type
      //    - returns by reference the element (or rather, element 
      //         cell) that this iterator points to
      reference operator*() { return ptr->element;  }

     
      // operator->
      //    - return type : a pointer to the element type
      //    - returns a pointer to the element that this
      //         iterator points to. 
      pointer operator->()  { return &(ptr->element); }

      
      // operator++ (prefix version)
      //    - return type : reference to an iterator
      //    -  
      iterator& operator++() { ptr= ptr->next; return *this; }

 
      // operator-- (prefix version)
      //    - return type: reference to an iterator     
      //    - 
      iterator& operator--()  { ptr = ptr->prev; return *this; }



      // operator++ (postfix version)
      //    - parameters : argument always integer 1, to distinguish 
      //                       from prefix version
      //    - return type : an iterator
      //    - 
      iterator operator++(int) 
      {
         iterator temp = *this;
         this->ptr = this->ptr->next;
         return temp;
      }


      // operator-- (postfix version)
      //    - parameters : argument always integer 1, to distinguish 
      //                       from prefix version
      //    - return type : an iterator
      //    - 
      iterator operator--(int) 
      {
         iterator temp = *this;
         this->ptr = this->ptr->prev;
         return temp;
      }



      // operator==
      //    - parameters : origVal - previously created iterator
      //    - return value : boolean integer
      //    - returns 1 if this iterator and the parameter 
      //         iterator point to the same location, 0 else
      int operator==(const iterator& origVal) const 
          { return (ptr == origVal.ptr); }


      // operator!=
      //    - parameters : origVal - previously created iterator
      //    - return value : boolean integer
      //    - returns 1 if this iterator and the parameter 
      //         iterator point different locations, 0 else
      int operator!=(const iterator& origVal) const
          { return (ptr != origVal.ptr); }



   private:

      friend class list<value_type>;
      friend class const_iterator;
      friend class reverse_iterator;
      friend class const_reverse_iterator;
      listNodePtr ptr;
      iterator(listNodePtr assignPtr) { ptr = assignPtr; } 
   };



   // *********************************************
   // *                                           *
   // *  list<>::const_iterator                   *
   // *                                           *
   // *  iterator for our list class. The code    *
   // *  should ideally be in the .C file, with   *
   // *  only the declarations in this file, but  *
   // *  the compiler (and indeed, most C++       *
   // *  compilers) do not yet support such an    *
   // *  arrangement for nested classes, so we    *
   // *  have the iterator code in this file.     *
   // *                                           *
   // *  Written July 1999 by Jason Zych          *
   // *                                           *
   // *********************************************

   class const_iterator
   {
   public:

      typedef Etype value_type;
      typedef Etype* const_pointer;
      typedef Etype& const_reference;

      // const_iterator
      //    - constructor
      //    - returns an iterator initialized to nothing
      const_iterator() { ptr = NULL;}



      // const_iterator
      //    - constructor
      //    - parameters : iter - an iterator
      //    - returns an iterator initialized to nothing
      const_iterator(const iterator& iter) {ptr = iter.ptr;}



      // operator*
      //    - return type : a reference to the element type
      //    - returns by reference the element (or rather, element 
      //         cell) that this iterator points to
      const_reference operator*() { return ptr->element;  }

     
      // operator->
      //    - return type : a pointer to the element type
      //    - returns a pointer to the element that this
      //         iterator points to. 
      const_pointer operator->()  { return &(ptr->element); }

      
      // operator++ (prefix version)
      //    - return type : reference to an iterator
      //    -  
      const_iterator& operator++() { ptr = ptr->next; return *this; }

 
      // operator-- (prefix version)
      //    - return type: reference to an iterator     
      //    - 
      const_iterator& operator--()  { ptr = ptr->prev; return *this; }



      // operator++ (postfix version)
      //    - parameters : argument always integer 1, to distinguish 
      //                       from prefix version
      //    - return type : an iterator
      //    - 
      const_iterator operator++(int) 
      {
         const_iterator temp = *this;
         this->ptr = this->ptr->next;
         return temp;
      }


      // operator-- (postfix version)
      //    - parameters : argument always integer 1, to distinguish 
      //                       from prefix version
      //    - return type : an iterator
      //    - 
      const_iterator operator--(int) 
      {
         const_iterator temp = *this;
         this->ptr = this->ptr->prev;
         return temp;
      }



      // operator==
      //    - parameters : origVal - previously created iterator
      //    - return value : boolean integer
      //    - returns 1 if this iterator and the parameter 
      //         iterator point to the same location, 0 else
      int operator==(const const_iterator& origVal) const 
          { return (ptr == origVal.ptr); }


      // operator!=
      //    - parameters : origVal - previously created iterator
      //    - return value : boolean integer
      //    - returns 1 if this iterator and the parameter 
      //         iterator point different locations, 0 else
      int operator!=(const const_iterator& origVal) const
          { return (ptr != origVal.ptr); }



   private:

      friend class list<value_type>;
      listNodePtr ptr;
      const_iterator(listNodePtr assignPtr) { ptr = assignPtr; } 
   };




   // ***********************************************
   // *                                             *
   // *  listr<>::reverse_iterator                  *
   // *                                             *
   // *  Reverse iterator for our list class. The   *
   // *  should ideally be in the .C file, with     *
   // *  only the declarations in this file, but    *
   // *  the compiler (and indeed, most C++         *
   // *  compilers) do not yet support such an      *
   // *  arrangement for nested classes, so we      *
   // *  have the iterator code in this file.       *
   // *                                             *
   // *  Written July 1999 by Jason Zych            *
   // *                                             *
   // ***********************************************

   class reverse_iterator
   {
   public:

      typedef Etype value_type;
      typedef Etype* pointer;
      typedef Etype& reference;



      // reverse_iterator
      //    - constructor
      //    - returns a reverse_iterator initialized to nothing
      reverse_iterator() { ptr = NULL; }



      // operator*
      //    - return type : a reference to the element type
      //    - returns by reference the element (or rather, element 
      //         cell) that this iterator points to
      reference operator*() 
      { 
        iterator temp = ptr;
        return *--temp;
      } 


      // operator->
      //    - return type : a pointer to the element type
      //    - returns a pointer to the element that this
      //         iterator points to. 
      pointer operator->() { return &(ptr->prev->element); }


      // operator++ (prefix version)
      //    - return type : reference to a reverse_iterator
      //    -  
      reverse_iterator& operator++() { ptr--; return *this; }



      // operator-- (prefix version)
      //    - return type: reference to a reverse_iterator     
      //    - 
      reverse_iterator& operator--() { ptr++; return *this; }



      // operator++ (postfix version)
      //    - parameters : argument always integer 1, to distinguish 
      //                       from prefix version
      //    - return type : a reverse_iterator
      //    - 
      reverse_iterator operator++(int)
      {
         reverse_iterator temp = *this;
         --ptr;
         return temp;
      }


      // operator-- (postfix version)
      //    - parameters : argument always integer 1, to distinguish 
      //                       from prefix version
      //    - return type : a reverse_iterator
      //    - 
      reverse_iterator operator--(int)
      {
         reverse_iterator temp = *this;
         ++ptr;
         return temp;
      }



      // operator==
      //    - parameters : origVal - previously created reverse_iterator
      //    - return value : boolean integer
      //    - returns 1 if this iterator and the parameter 
      //         iterator point to the same location, 0 else
      int operator==(const reverse_iterator& origVal) const
          { return (ptr == origVal.ptr); }


      // operator!=
      //    - parameters : origVal - previously created reverse_iterator
      //    - return value : boolean integer
      //    - returns 1 if this iterator and the parameter 
      //         iterator point different locations, 0 else
      int operator!=(const reverse_iterator& origVal) const
          { return (ptr != origVal.ptr); }


   private:

      friend class list<Etype>;
      friend const_reverse_iterator;
      iterator ptr;
      reverse_iterator(listNodePtr assignPtr) : ptr(assignPtr)  {};
      reverse_iterator(iterator assignIter) : ptr(assignIter) {}
   };



   // ***********************************************
   // *                                             *
   // *  list<>::const_reverse_iterator             *
   // *                                             *
   // *  Reverse iterator for our list class. The   *
   // *  should ideally be in the .C file, with     *
   // *  only the declarations in this file, but    *
   // *  the compiler (and indeed, most C++         *
   // *  compilers) do not yet support such an      *
   // *  arrangement for nested classes, so we      *
   // *  have the iterator code in this file.       *
   // *                                             *
   // *  Written July 1999 by Jason Zych            *
   // *                                             *
   // ***********************************************

   class const_reverse_iterator
   {
   public:

      typedef Etype value_type;
      typedef const Etype* const_pointer;
      typedef const Etype& const_reference;


      // const_reverse_iterator
      //    - constructor
      //    - returns a const_reverse_iterator initialized to nothing
      const_reverse_iterator() { ptr = NULL; }


      // const_reverse_iterator
      //    - constructor
      //    - parameters : iter - an iterator
      //    - returns an iterator initialized to nothing
      const_reverse_iterator(const reverse_iterator& iter) {ptr = iter.ptr;}


      // operator*
      //    - return type : a reference to the element type
      //    - returns by reference the element (or rather, element 
      //         cell) that this iterator points to
      const_reference operator*() 
      { 
        reverse_iterator temp = ptr;
        return *--temp;
      } 


      // operator->
      //    - return type : a pointer to the element type
      //    - returns a pointer to the element that this
      //         iterator points to. 
      const_pointer operator->() { return &(ptr->prev->element); }


      // operator++ (prefix version)
      //    - return type : reference to a reverse_iterator
      //    -  
      const_reverse_iterator& operator++() { ptr--; return *this; }



      // operator-- (prefix version)
      //    - return type: reference to a reverse_iterator     
      //    - 
      const_reverse_iterator& operator--() { ptr++; return *this; }




      // operator++ (postfix version)
      //    - parameters : argument always integer 1, to distinguish 
      //                       from prefix version
      //    - return type : a reverse_iterator
      //    - 
      const_reverse_iterator operator++(int)
      {
         const_reverse_iterator temp = *this;
         --ptr;
         return temp;
      }


      // operator-- (postfix version)
      //    - parameters : argument always integer 1, to distinguish 
      //                       from prefix version
      //    - return type : a reverse_iterator
      //    - 
      const_reverse_iterator operator--(int)
      {
         const_reverse_iterator temp = *this;
         ++ptr;
         return temp;
      }



      // operator==
      //    - parameters : origVal - previously created reverse_iterator
      //    - return value : boolean integer
      //    - returns 1 if this iterator and the parameter 
      //         iterator point to the same location, 0 else
      int operator==(const const_reverse_iterator& origVal) const
          { return (ptr == origVal.ptr); }

      // operator!=
      //    - parameters : origVal - previously created reverse_iterator
      //    - return value : boolean integer
      //    - returns 1 if this iterator and the parameter 
      //         iterator point different locations, 0 else
      int operator!=(const const_reverse_iterator& origVal) const
          { return (ptr != origVal.ptr); }


   private:

      friend class list<Etype>;
      iterator ptr;
      const_reverse_iterator(listNodePtr assignPtr) : ptr(assignPtr)  {};
      const_reverse_iterator(iterator assignIter) : ptr(assignIter) {}
   };








  // ************* Start of list class functions ******************


 // *** Constructors and Big Three

   // list
   //    - default constructor
   //    - creates an empty list
   list();



   // list
   //    - constructor
   //    - parameters : listSize - number of elements in this list
   //    - creates a list of listSize default elements   
   list(int listSize); 



   // list
   //    - constructor
   //    - parameters : listSize - number of elements in this list
   //    - creates a list of listSize elements each initialized to 
   //           initElem 
   list(int listSize, Etype initElem); 



/*
   This function requires that we further templatize this
   function...will CC allow that? For now I will leave it out;
   might add later...

   // list
   //    - constructor
   //    - parameters : first - first iterator of range
   //                 : last - last iterator of range
   //    - initializes a vector to have all elements in 
   //          range [first, last)
   list(iterator first, iterator last);

*/



   // list
   //    - copy constructor
   //    - parameters : origVal - a previously allocated list object
   //    - initializes the list to be a copy of origVal
   list(const list& origVal);




   // ~list
   //    - destructor
   //    - deallocates all dynamically allocated memory inside the list
   virtual ~list();



   // operator=
   //    - parameters: origVal - a previously allocated list object
   //    - return value: reference to the list object
   //    - sets the the list to be a copy of origVal
   list& operator= (const list& origVal);




   // assign
   //    - parameters : vectorSize - new size of vector
   //                 : assignElem - assigning element
   //    - assigns vector to be of given size with each cell
   //        assigned to hold assignElem
   void assign(int listSize, Etype assignElem);



/*

   This function requires that we further templatize this
   function...will CC allow that? For now I will leave it out;
   might add later...

   // assign
   //    - parameters : first - first iterator of range
   //                 : last - last iterator of range
   //    - assigns a vector to have all elements in
   //          range [first, last)
   void assign(iterator first, iterator last);

*/




 // *** list public functions for iterators

   // begin
   //    - return type : iterator
   //    - returns an iterator that points to first
   //        node of list
   iterator begin();


   // begin
   //    - return type : iterator
   //    - returns an iterator that points to first
   //        node of list
   const_iterator begin() const;


   // end
   //    - return type : iterator
   //    - returns an iterator that points to "after the last node"
   //        so we know we are "off the list".
   iterator end();


   // end
   //    - return type : iterator
   //    - returns an iterator that points to "after the last node"
   //        so we know we are "off the list".
   const_iterator end() const;


   // rbegin
   //    - return type : reverse iterator
   //    - returns a reverse iterator that points to
   //        first node of abstract "reverse list", i.e.
   //        last real node of list
   reverse_iterator rbegin();



   // rbegin
   //    - return type : reverse iterator
   //    - returns a reverse iterator that points to
   //        first node of abstract "reverse list", i.e.
   //        last real node of list
   const_reverse_iterator rbegin() const;

 
   // rend
   //    - return type : reverse iterator
   //    - returns a reverse iterator that points to
   //        "end position" of abstract "reverse list",
   //         i.e. the abstract "before first element"
   //         spot in list  
   reverse_iterator rend();
 

   // rend
   //    - return type : reverse iterator
   //    - returns a reverse iterator that points to
   //        "end position" of abstract "reverse list",
   //         i.e. the abstract "before first element"
   //         spot in list
   const_reverse_iterator rend() const;
 


 // ******** Element access


   // front
   //    - return type : a constant reference to a list element
   //    - returns a constant reference to the first cell in the list 
   const Etype& front() const;


   // front
   //    - return type : a reference to a list element 
   //    - returns a reference to the first cell in the list 
   Etype& front();


   // back 
   //    - return type : a constant reference to a list element 
   //    - returns a constant reference to the first cell in the list 
   const Etype& back() const;


   // back 
   //    - return type : a reference to a list element 
   //    - returns a reference to the first cell in the list 
   Etype& back();



 // ********** (Stack/Queue)-type access


   // push_back
   //    - parameters : insElem - element to insert
   //    - inserts element as last element in list   
   void push_back(Etype insElem);


   // pop_back
   //    - removes last element of list (but does NOT
   //        return it)
   void pop_back(); 
  

   // push_front
   //    - parameters : insElem - element to insert
   //    - inserts element as first element in list
   void push_front(Etype insElem);


   // pop_front
   //    - removes first element of list (but does NOT 
   //        return it. 
   void pop_front(); 



 // *************** General list operations


   // insert
   //    - parameters : insIter - an iterator to the element we want to
   //                     insert before
   //                 : newElem - an element of the list's type
   //    - inserts newElem before the position indicated by the iterator
   void insert(iterator insIter, Etype newElem); 



   // insert
   //    - parameters : insIter - an iterator to the element we want 
   //                     to insert before
   //                 : the number of copies of our new element that 
   //                     we want
   //                 : newElem - an element of the list's type
   //    - inserts numCopies copies of newElem before the position 
   //        indicated by the iterator
   void insert(iterator insIter, int numCopies, Etype insElem);



   // insert
   //    - parameters : insIter - an iterator to the element we want 
   //                     to insert before
   //                 : first - an iterator to the first element in
   //                     a sequence of elements we want to insert 
   //                 : last - an iterator to the element just past
   //                     the last element in the sequence we want 
   //                     to insert (i.e. we are inserting the values
   //                     in the range [first, last) meaning first
   //                     is included, but last is not
   //    - inserts the sequence first (included) through last 
   //        (non-included) into this list before the position 
   //        indicated by insIter
   void insert(iterator insIter, iterator first, iterator last);



   // erase 
   //    - parameters : eraseIter - iterator to element to be removed
   //    - removes the element at the current postition of the list
   void erase(iterator eraseIter);



   // erase
   //    - parameters : first - iterator to first element in sequence
   //                     to be removed
   //                 : last - iterator to element after last one to
   //                     be removed (see 3rd insert comments for more
   //                     explanation) 
   //    - removes the sequence [first, last) from this list
   void erase(iterator first, iterator last);  


   // clear
   //    - deletes all values from list, resulting in an empty list
   void clear();


// **************** Other general operations

   // size
   //    - parameters : none
   //    - return value : integer
   //    - returns size of vector 
   int size() const; 


   // Empty
   //    - parameters : none
   //    - return value : boolean integer
   //    - returns 1 if vector is empty, 0 else
   int empty() const;


   // operator==
   //    - parameters : secondVal - another list object to compare to
   //    - return type : boolean integer
   //    - returns 1 if this list is equal to secondVal, 0 else 
   int operator==(const list& secondVal);

 

   // operator!=
   //    - parameters : secondVal - another list object to compare to
   //    - return type : boolean integer
   //    - returns 1 if this list is not equal to secondVal, 0 else 
   int operator!=(const list& secondVal);


   // operator<
   //    - parameters : secondVal - another list object to compare to
   //    - return type : boolean integer
   //    - returns 1 if this list is less than secondVal, 0 else 
   int operator<(const list& secondVal);




// **************  list-class-specific operations


   // splice
   //    - parameters : insPosition - the place where we want
   //                     to splice the new list into
   //                 : newlist - the list which we will splice out
   //                     of and into this list
   //                 : first - iterator to first element in newlist
   //                        involved in the splice
   //                 : last - iterator to element in newlist just after
   //                     last element involved in the splice
   // - insert the segment of newlist between first and last -- including
   //     first but not last, i.e. the interval [first, last) -- into our
   //     own list before insPosition without copying -- i.e. remove
   //     them from newlist and splice them into our list 
   void splice(iterator insPosition, list& newlist, iterator first, 
                                                          iterator last);


   // merge
   //    - parameters : mergelist - the list we want to merge into 
   //                     this one
   //    - Assumes that this list and mergelist are already sorted.
   //    - Merge removes elements from mergelist and inserts them into this
   //        list while preserving element orders so that the final list turns 
   //        out to be sorted.  
   void merge(list& mergelist);


 
 
   // split
   // 	- parameters: pos - the position where we want to split the list
   //   - this list will be splitted into two list before pos, and the new 
   //		list will be from pos on.
   list<Etype> split(iterator pos);




   // operator<<
   //    - friend function of list class
   //    - parameters : Out - the output stream to which we will write
   //                 : printlist - the list that we will write to the 
   //                                                   output stream
   //    - return value : a reference to the same output stream we passed in
   //    - writes list values to parameter stream and returns stream
   friend std::ostream&
              operator<< <Etype> (std::ostream& Out, 
                                             const list<Etype>& printlist);



protected:

   friend class iterator;
   friend class const_iterator;


   class listNode
   {
   public:    // public within context of list class
              // protected outside list class

      // listNode Constructor
      listNode() { prev = NULL; next = NULL; }
      listNode(Etype elem) { prev=NULL; next = NULL; element = elem; }

      Etype element;
      listNode* next;
      listNode* prev;
   };


   listNode *headerNode;  

   int listSize;          // number of nodes in list


};

#endif 

   

