/*
 *  walker.cpp
 *  Active Walkers
 *
 *  Created by Shane Stafford on Wed Nov 27 2002.
 *  Copyright (c) 2002. All rights reserved.
 *
 */

#include <iostream>
#include "walker.h"
#include "array2d.h"
#include "math.h"

// name=setWalkerParams
//	-set parameters for all walker instances
void walker::setWalkerParams( double RepThresh, double InactThresh, 
                      double Uptake, double Metab, double Step, 
                      double EnergyInit, double EnergyReproduction,
                      int Hardness, array2d< int >* envLattice, 
                      array2d< double >* foodLattice )
{
    repThreshold = RepThresh; 
    inactThreshold = InactThresh; 
    maxUptake = Uptake; 
    metabolism = Metab;
    maxStep = Step;
    initEnergy = EnergyInit;
    repEnergy = EnergyReproduction;
    envHardness = Hardness - 1;
    envelope = envLattice;
    food = foodLattice;
}

// name=getInfo
//      -get printable parameters
void walker::getInfo( double& _x, double& _y, double& _energy )
{
  _x = x;
  _y = y;
  _energy = energy;
}


/*
   name=step
	-Public step function for walker.  Metabolizes, moves, eats, reproduces.
	-Returns an int and changes walker* with the following parameters:
		int = 0, walker* = 0:		succesful step
		int = 1, walker* = 0;		walker became inactive
		int = 3, walker* = new walker	walker reproduced; returns new walker*
*/
int walker::step( double rnd1, double rnd2, walker*& child )
{
    child = 0;
    
    //check if walker is inactive
    if( metabolize() )
        return 1;
    
    //perform step
    move( rnd1, rnd2 );

    //eat food from lattice
    eat();
    
    //reproduce, if possible
    child = reproduce();
    if( child ) {
        //a child was returned
        return 2;
    }
    //no reproduction
    return 0;
}
                            
// name=constructor
//	-public constructor
walker::walker( double _x, double _y ) : 
        x( _x ), 
        y( _y ),
        energy( initEnergy )
{
    return;
}

// name=constructor
//      -initial constructor
walker::walker( double _x, double _y, double _energy ) :
        x( _x ),
	y( _y ),
	energy( _energy )
{
  return;
}



// name=metabolize
//	-use energy and check to see if inactive
int walker::metabolize()
{
    energy -= metabolism;
    if(energy < inactThreshold)
        return 1;	//move to inactive walker list
    return 0;		//keep walking
}

// name=move
//	-takes in two random numbers that are mapped into direction and distance
//	-makes move if envelope is open; otherwise envelope is incremented and
//	 no move is made.
void walker::move( double rnd1, double rnd2 )
{
    double r = rnd1 * maxStep;
    double theta = rnd2 * 2 * 3.14159;

    double dblX = x + r * cos( theta );
    double dblY = y + r * sin( theta );
    
    int xCoord = (int)dblX;
    int yCoord = (int)dblY;
    if( (*envelope)( xCoord, yCoord ) < envHardness ) {
        //envelope not open yet; add 1 to hit count
        ( (*envelope)( xCoord, yCoord ) )++;
	return;
    }
    
    //envelope open; make move
    x = dblX;
    y = dblY;
    return;
}

// name=eat
//	-eat maximum amount possible from lattice point
void walker::eat()
{
    //get lattice points
    int xCoord = (int)x;
    int yCoord = (int)y;
    
    double available = (*food)( xCoord, yCoord );
    if( available > 0.0 ) {
        double deltaE = ( ( available > maxUptake ) ? maxUptake : available );
        energy += deltaE;
        (*food)( xCoord, yCoord ) -= deltaE;
    }
}

// name=reproduce
//	-if possible, reproduces and returns a new walker
walker* walker::reproduce()
{
    //make sure threshold has been reached
    if( energy < repThreshold )
        return 0;
    
    //reduce energy by child's initialization + reproduction energy
    energy -= (repEnergy + initEnergy);
    
    //initialize child walker at same location
    // .... if the child is added to the end of the list, it will be stepped a random
    //      distance away from parent in the same iteration
    walker* child = new walker( x, y );
    
    return child;
}


// initialization list
double walker::repThreshold = 1;		//reproduction threshold
double walker::inactThreshold = 0;		//unactivate threshold
double walker::metabolism = 0.0667;		//energy usage rate
double walker::maxUptake = 0.2;		        //maximum feeding rate
double walker::maxStep = 0.2;			//largest step size
double walker::initEnergy = 0.33;		//initial energy
double walker::repEnergy = 0.3;			//energy used for reproduction
int walker::envHardness = 5;			//envelope hardness in # of hits
array2d< int >* walker::envelope = 0;		//envelope lattice
array2d< double >* walker::food = 0;		//food lattice

