
// Solve a linear system using LU factorization routines of Lapack.
//
// Usage: a.out < matrix.dat
//
// where matrix.dat is an ASCII file consisting of the
// matrix size (M,N) followed by its values.  For example,
//
//  3  2
//  8.1  1.2  4.3
//  1.3  4.3  2.9
//
//
// Compile line (on g++/g77 systems) looks like
//
// g++ foo.cc -llapack -lblas -lf2c -lm
//



#include <iostream.h>

#include "tnt.h"
#include "fortran.h"
#include "lapack.h"
#include "lu.h"

using namespace TNT;
using namespace std;

int main()
{
    Fortran_Matrix<double> A;

    cin >> A;


    Subscript N = A.num_rows();
    Subscript M = A.num_cols();
    assert(M == N);


    Vector<double> b(N, 1.0);   // b= [1,1,1,...]
    Vector<Subscript> index(N);
    Vector<double> x = b;

    cerr << "Original Matrix A: " << A << endl;
    
    Fortran_Matrix<double> B(A);

    if (LU_factor(B, index) !=0)
    {
        cerr << "LU_factor() failed." << endl;
        exit(1);
    }

    cerr << "index: " << index << endl;

    if (LU_solve(B, index, x) != 0)
    {
        cerr << "LU_Solve() failed." << endl;
        exit(1);
    }
    cerr << "Solution x for Ax=b, where b=[1,1,...] " <<endl;
    cerr << " x: " << x << endl;

    cerr << "A*x should be the vector [1,1,...] "  <<endl;
    cerr     << "residual [A*x - b]: " << A*x  - b << endl;
    

    // Now  solve it with LU factorization in Lapack

    x = Lapack_LU_linear_solve(A,b);

    cerr << " x (using LU Lapack): " << x << endl;
    cerr     << "residual [A*x - b]: " << A*x  - b << endl;


    // Now solve it with QR factorization in Lapack

    x = Lapack_LLS_QR_linear_solve(A, b);

    cerr << " x (using QR Lapack): " << x << endl;
    cerr     << "residual [A*x - b]: " << A*x - b << endl;

	return 0;
    
}
