#include "md.h"

int main(int argc, char** argv)
{
    char *filename, *restore;
    double T,sigma,L,z,eta,gam_dot,gam_eff,A,lamx,lamy,dk,layer_size;
    double mnorm, dt, rc_wall, eps, eps_w,dtbase;
    int kbins,maxkbin,N,maxlag,t_init,rseed,t,tmax,datacount,currlag;
    int num_layers, i;
    Atom3D_t *p;
    Vector3D_t **vmat, *KE;
    double *rho, *vac;
    Scalar_Avg_t *S, *vz;
   

    kbins=25;
    rc_wall=2.5;
    maxlag=1000;
    switch(argc)
    {
      case 2:
        {
            filename=argv[1];
            Read_Input(filename,&T,&sigma,&L,&z,&eta,&gam_dot,&A,&lamx,&lamy,&eps,&eps_w,&layer_size,&rseed,&tmax,&datacount);
            dk=2*PI/kbins;
            maxkbin=KMAX/dk;
            Initialize_Program(&N,L,z,A,eta,&p,&vz,&rho,&S,&vmat,&vac,maxlag,maxkbin,layer_size,&num_layers,&KE,tmax);
            for(i=0;i<N;i++)
            {
                p[i].mass=1.0;
                p[i].sigma=sigma;
            }
            mnorm=sqrt(T*eps/1.0);
            Initialize_System(sigma,eta,L,z,A,N,mnorm,p);
            t_init=0;
            srand48(rseed);
            break;
        }

      case 3:
        {
            filename=argv[1];
            restore=argv[2];
            Read_Input(filename,&T,&sigma,&L,&z,&eta,&gam_dot,&A,&lamx,&lamy,&eps,&eps_w,&layer_size,&rseed,&tmax,&datacount);
            dk=2*PI/kbins;
            maxkbin=KMAX/dk;
            Initialize_Program(&N,L,z,A,eta,&p,&vz,&rho,&S,&vmat,&vac,maxlag,maxkbin,layer_size,&num_layers,&KE,tmax);
            Read_Continue(p,N,filename,vmat,&maxlag,&t);
            t_init=t;
            srand48(rseed);
            break;
        }
      default:
        {
            usage(argv[0]);
            exit(0);
        }
    }

/*  This is the main loop of the program.  Here's where we actually do
    the simulation. */
    printf("%d particles in system.\n",N);
    dtbase=sqrt(eps/(sigma*sigma));
    for(t=t_init;t<tmax;t++)
    {
        dt=0.00000001*dtbase;
/*        dt=MIN((t/maxlag),1)*dtbase;*/
        gam_eff=MIN((t/maxlag),1)*gam_dot;
        Predict_System(p,dt,N,L,z,A);
        Calc_Forces(p,rc_wall,A,lamx,lamy,gam_eff,dt,N,eps_w,eps,z);
        KE[t]=Correct_System(p,dt,N,L,z,A);
        /* Copy the velocities over to the correct place in the velocity
           matrix. */
        for(i=0;i<N;i++)
        {
            vmat[i][t%maxlag].x=p[i].v.x;
            vmat[i][t%maxlag].y=p[i].v.y;
            vmat[i][t%maxlag].z=p[i].v.z;
        }

        if(0==t%datacount)
	{
            printf("Calculating data.\n");
            if(t<maxlag)
            currlag=t;
            else
            currlag=maxlag;
            Write_Energy(KE,tmax,filename);
            Calc_Vz(layer_size,z,N,p,vz,A,num_layers);
            printf("After Vz\n");
            Calc_Sk(L,z,S,maxkbin,dk,N,p);
            printf("After Sk\n");
            Calc_Vac(vmat,vac,currlag,maxlag,t,N);
            printf("After Vac\n");
            Write_Output(S,vac,vz,maxkbin,maxlag,num_layers,dk,layer_size,t,filename);
            printf("After output.\n");
            Write_Continue(p,N,filename,vmat,maxlag,t);
            printf("Current iteration: %d\n",t);
	}

    }
    Write_Energy(KE,tmax,filename);
    currlag=maxlag;
    Calc_Vz(layer_size,z,N,p,vz,A,num_layers);
    Calc_Sk(L,z,S,maxkbin,dk,N,p);
    Calc_Vac(vmat,vac,currlag,maxlag,t,N);
    Write_Output(S,vac,vz,maxkbin,maxlag,num_layers,dk,layer_size,t,filename);
    Write_Continue(p,N,filename,vmat,maxlag,t);
    Cleanup(p,vz,rho,S,vmat,vac,N);
    exit(0);
}


/*  Gear 5 value predictor/corrector integrator.  */
void Predict_System(Atom3D_t* p, double dt, int N, double L, double z, double A)
{
    double c1, c2, c3, c4;
    int i;
    double half_L, inv_L;

    c1=dt;
    c2=c1*dt/2.0;
    c3=c2*dt/3.0;
    c4=c3*dt/4.0;

    half_L=L/2.0;
    inv_L=1.0/L;

    for(i=0;i<N;i++)
    {
        /*  Taylor expansion around the positions. */
        p[i].U.x=p[i].U.x+c1*p[i].v.x+c2*p[i].a.x+c3*p[i].b.x+c4*p[i].c.x;
        p[i].U.y=p[i].U.y+c1*p[i].v.y+c2*p[i].a.y+c3*p[i].b.y+c4*p[i].c.y;
        p[i].U.z=p[i].U.z+c1*p[i].v.z+c2*p[i].a.z+c3*p[i].b.z+c4*p[i].c.z;
        /*  Taylor expansion around the velocities. */
        p[i].v.x=p[i].v.x+c1*p[i].a.x+c2*p[i].b.x+c3*p[i].b.x;
        p[i].v.y=p[i].v.y+c1*p[i].a.y+c2*p[i].b.y+c3*p[i].b.y;
        p[i].v.z=p[i].v.z+c1*p[i].a.z+c2*p[i].b.z+c3*p[i].b.z;
        /*  Taylor expansion around the accelerations. */
        p[i].a.x=p[i].a.x+c1*p[i].b.x+c2*p[i].c.x;
        p[i].a.y=p[i].a.y+c1*p[i].b.y+c2*p[i].c.y;
        p[i].a.z=p[i].a.z+c1*p[i].b.z+c2*p[i].c.z;
        /*  Taylor expansion around the acceleration derivatives. */
        p[i].b.x=p[i].b.x+c1*p[i].c.x;
        p[i].b.y=p[i].b.y+c1*p[i].c.y;
        p[i].b.z=p[i].b.z+c1*p[i].c.z;
	p[i].R.x=p[i].U.x-L*floor(((p[i].U.x+half_L)*inv_L));
	p[i].R.y=p[i].U.y-L*floor(((p[i].U.y+half_L)*inv_L));
	p[i].R.z=p[i].U.z;

        if((p[i].U.x>half_L)||(p[i].U.x<-(half_L)))
        printf("X coordinates SHOULD be folded.\n");
        if(p[i].U.x!=p[i].R.x)
        printf("X coordinates folded from %lf to %lf.\n",p[i].U.x,p[i].R.x);

        if((p[i].U.y>half_L)||(p[i].U.y<-(half_L)))
        printf("Y coordinates SHOULD be folded.\n");
        if(p[i].U.y!=p[i].R.y)
        printf("Y coordinates folded from %lf to %lf.\n",p[i].U.y,p[i].R.y);

        if(fabs(p[i].R.z)>(z/2+A))
        {
            printf("Error:  Particle %d beyond the (predictor) wall!!!\n",i);
            printf("Particle %d position: %lf\n",i,p[i].R.z);
        }
    }
    return;
}

Vector3D_t Correct_System(Atom3D_t* p, double dt, int N, double L, double z, double A)
{

    double gear0,gear1,gear3,gear4;
    double c1,c2,c3,c4;
    double corr_r,corr_v,corr_b,corr_c;
    double half_L, inv_L;
    Vector3D_t a, corr, KE;
    int i;
    /*  This routine is converted from the code presented on the website
        referenced in Allen and Tildesley's "Computer Simulation of Liquids."
        The only change made has been to remove the masses in the initial
        acceleration calculation from the forces, since we will ALWAYS be
        working in homogeneous systems with a fundamental mass taken to be
        1. */

    gear0=19.0/120.0;
    gear1=3.0/4.0;
    gear3=1.0/2.0;
    gear4=1.0/12.0;

    c1=dt;
    c2=c1*dt/2.0;
    c3=c2*dt/3.0;
    c4=c3*dt/4.0;

    corr_r=gear0*c2;
    corr_v=gear1*c2/c1;
    corr_b=gear3*c2/c3;
    corr_c=gear4*c2/c4;

    half_L=L/2.0;
    inv_L=1.0/L;
    KE.x=0.0;
    KE.y=0.0;
    KE.z=0.0;
    for(i=0;i<N;i++)
    {
        /*  Calculate the acceleration. */
        a.x=p[i].F.x;
        a.y=p[i].F.y;
        a.z=p[i].F.z;
        /*  Calculate the correction factors. */
        corr.x=a.x-p[i].a.x;
        corr.y=a.y-p[i].a.y;
        corr.z=a.z-p[i].a.z;
        /*  Calculate the corrected (unfolded) positions. */
        p[i].U.x=p[i].U.x+corr_r*corr.x;
        p[i].U.y=p[i].U.y+corr_r*corr.y;
        p[i].U.z=p[i].U.z+corr_r*corr.z;
        /*  Calculate the corrected velocities. */
        p[i].v.x=p[i].v.x+corr_v*corr.x;
        p[i].v.y=p[i].v.y+corr_v*corr.y;
        p[i].v.z=p[i].v.z+corr_v*corr.z;
        /*  Assign the corrected accelerations. */
        p[i].a.x=a.x;
        p[i].a.y=a.y;
        p[i].a.z=a.z;
        /*  Calculate the corrected acceleration derivatives. */
        p[i].b.x=p[i].b.x+corr_b*corr.x;
        p[i].b.y=p[i].b.y+corr_b*corr.y;
        p[i].b.z=p[i].b.z+corr_b*corr.z;
        /*  Calculate the corrected acceleration second derivatives. */
        p[i].c.x=p[i].c.x+corr_c*corr.x;
        p[i].c.y=p[i].c.y+corr_c*corr.y;
        p[i].c.z=p[i].c.z+corr_c*corr.z;
	/*  Calculate the folded positions. */
	p[i].R.x=p[i].U.x-L*floor(((p[i].U.x+half_L)*inv_L));
	p[i].R.y=p[i].U.y-L*floor(((p[i].U.y+half_L)*inv_L));
	p[i].R.z=p[i].U.z;

        KE.x+=p[i].v.x*p[i].v.x;
        KE.y+=p[i].v.y*p[i].v.y;
        KE.z+=p[i].v.z*p[i].v.z;
        if((p[i].U.x>half_L)||(p[i].U.x<-(half_L)))
        printf("X coordinates SHOULD be folded.\n");
        if(p[i].U.x!=p[i].R.x)
        printf("X coordinates folded from %lf to %lf.\n",p[i].U.x,p[i].R.x);

        if((p[i].U.y>half_L)||(p[i].U.y<-(half_L)))
        printf("Y coordinates SHOULD be folded.\n");
        if(p[i].U.y!=p[i].R.y)
        printf("Y coordinates folded from %lf to %lf.\n",p[i].U.y,p[i].R.y);

        if(fabs(p[i].R.z)>(z/2+A))
        {
            printf("Error:  Particle %d beyond the (corrector) wall!!!\n",i);
            printf("Particle %d position: %lf\n",i,p[i].R.z);
            exit(0);
        }
        KE.x*=.5;
        KE.y*=0.5;
        KE.z*=0.5;
        
    }
    return(KE);
}

void usage(char *progname)
{
    printf("Usage:\n\t\t%s configuration file [continue file]\n\n",progname);
    exit(0);
}













