#include "md.h"

/*  Calculation routines for various quantities. */

void Calc_Vz(double layer_size, double z, int N, Atom3D_t* p, Scalar_Avg_t* v_profile, double A, int num_layers)
{
    /*  This subroutine calculates the velocity profile perpendicular to the
        shearing walls. */

    int index, i;
    double half_z;

    half_z=z/2+A;
    /*  First initialize out the velocity profile. */
    for(i=0;i<num_layers;i++)
    {
        v_profile[i].val=0.0;
    }
    
    /*  Now for each atom place it somewhere within a layer, storing its
        velocity vectors. */
    for(i=0;i<N;i++)
    {
        index=(int) ((p[i].R.z+half_z)/layer_size);
        if((index<0)||(index>num_layers))
        printf("Velocity index out of range.\n");
        v_profile[index].val+=p[i].v.x;
        v_profile[index].n++;
    }

    /*  And average the velocities once we're done. */
    for(i=0;i<num_layers;i++)
    {
        if(v_profile[i].n!=0)
        v_profile[i].val=v_profile[i].val/v_profile[i].n;
    }
    return;
}

void Calc_Sk(double L, double z, Scalar_Avg_t* S, int maxind, double dk, int N, Atom3D_t* p)
{

    int vec_skip,k_int,kx,ky,kz,i,j;
    double per_K,z_K,K_mag,KdotRi,KdotRj,sinkri,sinkrj,coskri,coskrj;
    Vector3D_t K;

  /*  This is the minimum K multiplier for the periodic lengths. */
  per_K=2*PI/L;
  /*  This is the minimum K multiplier for the wall vectors. */
  z_K=2*PI/z;

  /*  Set this number to skip to every nth K vector.. this is a 
      rather nasty computation. */
    vec_skip=10;
    
    /*  Initialize the S[k] array. */
  for(i=0;i<maxind;i++)
  {
      S[i].val=0.;
      S[i].n=0;
  }
  
  /*  Loop through possible K vectors. */
  for(kx=1;kx<maxind;kx+=vec_skip)
    for(ky=1;ky<maxind;ky+=vec_skip)
      for(kz=1;kz<maxind;kz+=vec_skip)
	{
            /*  Find the specific K vector we're working with. */
            K.x=kx*per_K;
            K.y=ky*per_K;
            K.z=kz*z_K;
            /*  Find the magnitude of the K vector we're working with.
                This is the value we use for k in S(k). */
            K_mag=sqrt(K.x*K.x+K.y*K.y+K.z*K.z);
            /*  Find which bin in S[k] this vector corresponds to. */
            k_int=(int) K_mag/dk;
            /*  If our K vector is smaller than the max we want to look
                at AND it's not the zero vector (i=j), go ahead and do
                the calculation. */
            if(((K_mag)!=0.)&&(K_mag<=KMAX))
	    {
                /*  We use the formula S(k)=<rho(k)rho(-k)>.  This 
                    simplifies into

      +------ Align these marks to see the formula. 
      V       Note:  (A.B) is the scalar product.

      |               N   N
      |             --- ---  _                                   _
      |           1 \   \   |                                     |
      | S(k)= 1+ --- |   |  |cos(K.Ri)cos(K.Rj)+sin(K.Ri)sin(K.Rj)|
      |           N /   /   |_                                   _|
      |             --- ---
      |               i!=j 
                */

                /*  This is the number of different times we calculate
                    this value of k (NOT the number of atoms we find
                    for this value of k.  */
                S[k_int].n++;

                for(i=0;i<(N-1);i++)
		{
                    /*  For each Ri, find (K.Ri), Sin(K.Ri), Cos(K.Ri)
                        Then...  */
                    KdotRi=K.x*p[i].R.x+K.y*p[i].R.y+K.z*p[i].R.z;
                    sinkri=sin(KdotRi);
                    coskri=cos(KdotRi);
                    /*  Step through Rj... */

                    for(j=(i+1);j<N;j++)
		    {
                        /*  Same here. */
                        KdotRj=K.x*p[j].R.x+K.y*p[j].R.y+K.z*p[j].R.z;
                        sinkrj=sin(KdotRi);
                        coskrj=cos(KdotRj);
                        /*  Now, we add not only the result of Ri composited 
                            with Rj, but also Rj composited with Ri, since 
                            there's no reason we HAD to pick i first. */
                        S[k_int].val+=2*(coskri*coskrj+sinkri*sinkrj);
		    }
		}
	    }
	}
    for(i=0;i<maxind;i++)
    S[i].val=S[i].val/N+1.;
    return;
}

void Calc_Rho(Atom3D_t* p, double* rho, int N, int rhobins, double drho, double L, double z)
{
    int i,j, ind;
    Vector3D_t dr;
    double rij, rmax, r_out, r_in, dV;

    if(z>L)
    rmax=z/2.;
    else
    rmax=L/2.;
    /*  First off, let's clear out any previous densities we might
        have.  */
    for(i=0;i<rhobins;i++)
    rho[i]=0.0;
    
    /*  Since we don't care which particle a specific density profile
        is calculated for and we only want the average density profile
        for all particles, we can use the fact that a hit for rho(|ij|)
        is also a hit for rho(|ji|) to cut the calculations in half. */
    for(i=0;i<(N-1);i++)
    for(j=(i+1);j<N;j++)
    {
        dr.x=p[j].R.x-p[i].R.x;
        dr.y=p[j].R.y-p[i].R.y;
        dr.z=p[j].R.z-p[i].R.z;
        rij=sqrt(dr.x*dr.x+dr.y*dr.y+dr.z*dr.z);

        /*  
           Due to the fact that our unit cell is a parallelepiped, but the
           density is assumed isotropic, it *IS* possible to find a radius
           for the density that's larger than the max radius we want to
           calculate for.  In a unit cube, this would correspond to 
           calculating the density in a radius outside the embedded unit
           sphere. 
           
           Also note that since we can specify the Z dimension
           independently of the other dimensions, we technically have all
           sorts of anisotropic features going on.  We'll gloss over those
           by assuming that Z/L isn't TOO small, and our system is 
           essentially isotropic.  If this isn't the case, these
           calculations aren't done in the correct symmetry basis, and are
           totally messed up.  */
        if(rij<=rmax)
        {
            ind=rij/drho;
            rho[ind]+=2.;
        }
    }


    /*  Here we normalize rho.  Generally, for a single rho profile we
        would divide by the volume element of that shell.  Since we've
        accumulated ALL the profiles together, however, we also need to
        divide by the number of samples we've taken, in this case N. */
    for(i=0;i<rhobins;i++)
    {
        r_out=(i+1)*drho;
        r_in=i*drho;
        dV=(4/3)*PI*(r_out*r_out*r_out-r_in*r_in*r_in);
        rho[i]=rho[i]/(N*dV);
    }
    return;
}   

void Calc_Vac(Vector3D_t** v_mat, double* vac, int curr_lag, int max_lag, int t, int N)
{
    
    int lag,i,j;
    double v0x, v0y, v0z, vtx, vty, vtz;
 
    if(curr_lag<max_lag)
    lag=curr_lag;
    else
    lag=max_lag;
    
    for(i=0;i<lag;i++)
    {
        vac[i]=0.0;
        for(j=0;j<N;j++)
        {
            /*  For clarity's sake, let's find v0(x,y,z) and vt(x,y,z) 
                separately before taking the dot product. */
            v0x=v_mat[j][t%max_lag].x;
            vtx=v_mat[j][(t+i)%max_lag].x;
            v0y=v_mat[j][t%max_lag].y;
            vty=v_mat[j][(t+i)%max_lag].y;
            v0z=v_mat[j][t%max_lag].z;
            vtz=v_mat[j][(t+i)%max_lag].z;

            /*  Here we actually accumulate the scalar dot product. */ 
            vac[i]+=v0x*vtx+v0y*vty+v0z*vtz;
	}
	/*  In addition, we'll normalize everything by <v0^2>. */
	vac[i]=vac[i]/vac[0];
    }
    return;
}




































