/*
 *  ADI_Solver.cpp
 *  Active Walkers
 *
 *  Created by Shane Stafford on Tue Nov 26 2002.
 *  Copyright (c) 2002. All rights reserved.
 *
 */

#include "ADI_Solver.h"
#include "ThomasAlg.h"
#include "array2d.h"

void SolvePDE(  array2d< double >& T, 	//initialized dependent variable 2-D matrix
                double lambda, 		//lambda = k * dT/ (dX)^2 where dX = dY
                int size,		//NxN matrix, size = N
                int steps )		//iterations
{
    double *RHS, *tempT;
    array2d< double > LHS(size, 3);
    array2d< double > newT(size, size);
    RHS = new double[size];
    tempT = new double[size];
    int i, j, nextI, lastI, nextJ, lastJ;
    
    double adjCoeff = -lambda;
    double baseCoeff = 2 * ( 1 + lambda);
    
    //loop over steps
    for(int step = 1; step <= steps; step++) {
        
        //*********************** Column tridiagonals ***********************
        //set up tridiagonal for each column for first half step
        for(i = 0; i<size; i++) {
            
            nextI = i+1;  lastI = i-1;
            if(i==0) lastI = nextI;
            else if(i==size-1) nextI = lastI;
                 
            for(j = 0; j<size; j++) {
            
                //correct for derivative BC's
                nextJ = j+1;  lastJ = j-1;
                if(j==0) lastJ = nextJ;
                else if(j==size-1) nextJ = lastJ;
                
                //compute coefficients and RHS
                LHS(j,0) = adjCoeff;
                LHS(j,1) = baseCoeff;
                LHS(j,2) = adjCoeff;
                RHS[j] = lambda * T(lastI,j) + 2 * (1 - lambda) * T(i,j) 
                         + lambda * T(nextI,j);
            }//j
            
            //correct first and last row coefficients
            LHS(0,0) = 0;
            LHS(0,2) *= 2;
            LHS(size-1,2) = 0;
            LHS(size-1,0) *= 2;
            
            //solve tridiagonal using Thomas Algorithm
            tridiagSolve(LHS, RHS, tempT, 0, size-1);
            
            //substitute result into newT
            for(j = 0; j<size; j++)
                newT(i,j) = tempT[j];
        }//i
        
        //substitute first half step into T
        for(i = 0; i<size; i++)
            for(j = 0; j<size; j++)
                T(i,j) = newT(i,j);
        //*********************** End Column tridiagonals *******************
        
        //*********************** Row tridiagonals **************************        
        //set up tridiagonal for each row for last half step
        for(j = 0; j<size; j++) {
            
            nextJ = j+1;  lastJ = j-1;
            if(j==0) lastJ = nextJ;
            else if(j==size-1) nextJ = lastJ;
            
            for(i = 0; i<size; i++) {
            
                //correct for derivative BC's
                nextI = i+1;  lastI = i-1;
                if(i==0) lastI = nextI;
                else if(i==size-1) nextI = lastI;
                
                //compute coefficients and RHS
                LHS(i,0) = adjCoeff;
                LHS(i,1) = baseCoeff;
                LHS(i,2) = adjCoeff;
                RHS[i] = lambda * T(i,lastJ) + 2 * (1 - lambda) * T(i,j) 
                         + lambda * T(i,nextJ);
            }//i
            
            //correct first and last row coefficients
            LHS(0,0) = 0;
            LHS(0,2) *= 2;
            LHS(size-1,2) = 0;
            LHS(size-1,0) *= 2;
            
            //solve tridiagonal using Thomas Algorithm
            tridiagSolve(LHS, RHS, tempT, 0, size-1);
            
            //substitute result into newT
            for(i = 0; i<size; i++)
                newT(i,j) = tempT[i];
        }//j
        
        //substitute last half step into T
        for(i = 0; i<size; i++)
            for(j = 0; j<size; j++)
                T(i,j) = newT(i,j);
        //*********************** Row tridiagonals **************************  
                
    }//steps

    delete RHS;
    delete tempT;

}//SolvePDE
