#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include "string.h"
#include "ADI_Solver.h"
#include "array2d.h"
#include "walker.h"
#include "list225.h"

FILE* openIt( char *, char * );
int readParam( FILE*, double & );
int readParam( FILE*, int & );
int readInoc( FILE *, int &, double &, double & );


int main (int argc, const char * argv[]) {
    // header
    std::cout << "Bacterial Simulations\n";
 
    //general parameters
    int size           = 200;
    int initWalkers    = 20;
    int totalSteps     = 2000;
    int diffusionSteps = 1;
    int seed           = 11;
    double peptoneConc = 5.0;
    double lambda      = 300 * 300 / (250 * 250);	//lambda = D * dt / dx**2 (unitless)

    //walker parameters
    double reproThresh = 1.0;
    double inactThresh = 0.0;
    double maxUptake   = 0.2;
    double metabolism  = 0.0667;
    double maxJump     = 0.4;
    double initEnergy  = 0.33;
    double reproEnergy = 0.30;
    int envelHits      = 6;
    
    double rnd1, rnd2;
    double curr_x = 0, curr_y = 0, curr_energy = 0;

    //get input file name
    char filename[256];
    if( argc > 1 )
      strcpy( filename, argv[1] );
    else {
      printf( "Enter the filename without the .param or .inoc extension: " );
      scanf( "%s", filename );
    }
    std::cout << "The output file will be " << filename << ".coord\n";
    
    //make filename
    char paramFile[256];
    strcpy( paramFile, filename );
    strncat( paramFile, ".param", 6 );
    
    //open parameter file and read parameters
    FILE* infile;
    if( (infile = openIt( paramFile, "rt" ) ) == NULL ) return 1;
    if( readParam( infile, size           	) ) return 1;
    if( readParam( infile, initWalkers    	) ) return 1;
    if( readParam( infile, totalSteps     	) ) return 1;
    if( readParam( infile, diffusionSteps 	) ) return 1;
    if( readParam( infile, seed           	) ) return 1;
    if( readParam( infile, peptoneConc    	) ) return 1;
    if( readParam( infile, lambda         	) ) return 1;
    if( readParam( infile, reproThresh     	) ) return 1;
    if( readParam( infile, inactThresh		) ) return 1;
    if( readParam( infile, maxUptake 		) ) return 1;
    if( readParam( infile, metabolism 		) ) return 1;
    if( readParam( infile, maxJump		) ) return 1;
    if( readParam( infile, initEnergy		) ) return 1;
    if( readParam( infile, reproEnergy		) ) return 1;
    if( readParam( infile, envelHits 		) ) return 1;
    fclose( infile );

    //initialize peptone and agar lattices
    array2d< double > peptone( size, size );
    array2d< int > agar( size, size );
    peptone.initialize( peptoneConc );
    agar.initialize( 0 );
   
   
    /*************************************************************************
     *                           inoculate lattice                           *
     *************************************************************************/
    std::cout << "Inoculating...\n";
    
    //this is the inoculation
    list< walker* > ActiveList;
    walker* tempPtr1, *tempPtr2;
    
    //make filename
    char inocFile[256];
    strcpy( inocFile, filename );
    strncat( inocFile, ".inoc", 5 );
    
    //open inoculation file and read initial walkers
    if( (infile = openIt( inocFile, "rt" ) ) == NULL ) return 1;
    int numInoc = 0, flag = 0;
    
    do {
        flag = readInoc( infile, numInoc, curr_x, curr_y );
        if( !flag ) {
            for(int i = 0; i < numInoc; i++) {
                //allocate new walker
                tempPtr1 = new walker( curr_x, curr_y, initEnergy );
                ActiveList.push_back( tempPtr1 );
            }
            //open envelope location
            agar( (int)curr_x, (int)curr_y ) = envelHits + 1;
        }
    } while( !flag );
   
    fclose( infile );
    
    //set walker parameters
    tempPtr1->setWalkerParams( reproThresh,       // reproductive threshold (energy unit)
				 inactThresh,       // inactive threshold
				 maxUptake,         // max uptake
				 metabolism,        // metabolism
				 maxJump,           // max jump
				 initEnergy,        // initial energy
				 reproEnergy,       // energy lost during reproduction
				 envelHits,         // hits to lubricate an envelope square
				 &agar,             // envelope lattice
				 &peptone );        // food lattice

    /**************************** end inoculation ****************************/


    //inactive list is empty
    list< walker* > SporeList;

    //initialize random number generator
    std::cout << "Initializing random number generator...\n";
    srand( seed );

    /************************************************************************
     *                            loop timesteps                            *
     ************************************************************************/
    std::cout << "Looping over timesteps...\n";
    //loop over timesteps
    int stepReturn;
    
    //iterators
    list< walker* >::iterator ActiveIter = ActiveList.begin();
    list< walker* >::iterator ActiveEnd = ActiveList.end();

    for(int i=0; i<totalSteps; i++) {
      
      //step walkers 
      ActiveIter = ActiveList.begin();
      while( ActiveIter != ActiveEnd ) {

	//check every 50 steps to see if walkers are getting close to edge
	//... alternatively, could just keep stepping until they do get close...
	if( i % 50 == 49 ) {
	  (*ActiveIter)->getInfo( curr_x, curr_y, curr_energy );
	  if( curr_x > 0.9*size || curr_x < 0.1*size || curr_y > 0.9*size || curr_y < 0.1*size ) {
	    std::cout << "Terminating because walker is nearing lattice boundary!\n";
	    totalSteps = 0;
	  }
	}

	rnd1 = rand() / (RAND_MAX + 0.01);
	rnd2 = rand() / (RAND_MAX + 0.01);
	stepReturn = (*ActiveIter)->step( rnd1, rnd2, tempPtr1 );
	
	//step and check for children (or sporulation)
	if( stepReturn == 1 ) {
	  //sporulation
	  SporeList.push_back( *ActiveIter );  //add to spore list
	  ActiveList.erase( ActiveIter++ );   //remove from active list
	}
	else if( stepReturn == 2 ) {
	  //reproduction
	  ActiveList.push_back( tempPtr1 );    //add child to active list
	  ActiveIter++;
	}
	else
	  //step with no other action
	  ActiveIter++;       
      }

      //solve diffusion lattice
      // std::cout << "Solving diffusion lattice...\n";
      SolvePDE( peptone, lambda, size, diffusionSteps );

    }
    /************************** end timestep loop ***********************/

    //inactive list iterators
    list< walker* >::iterator SporeIter = SporeList.begin();
    list< walker* >::iterator SporeEnd = SporeList.end();

    /********************************************************************
     *                          write output                            *
     ********************************************************************/
    std::cout << "Writing output to file...\n";
    //make filename
    char coordFile[256];
    strcpy( coordFile, filename );
    strncat( coordFile, ".coord", 6 );
    
    //open coord file and write walker coordinates and energy
    FILE* outfile;
    if( (outfile = openIt( coordFile, "w" ) ) != NULL ) {
        
        //print header
        fprintf(outfile,"#Active walker algorithm test\n#\n# %8s %8s %8s\n", "X", "Y", "Energy");
    
        //print output
        int numActive = 0;
        int numSpores = 0;
        //active list
        ActiveIter = ActiveList.begin();
        while( ActiveIter != ActiveEnd ) {
            (*ActiveIter)->getInfo( curr_x, curr_y, curr_energy );
            fprintf(outfile, "%8.2f %8.2f %8.4f\n", curr_x, curr_y, curr_energy );
            ActiveIter++; 
            numActive++;
        }
    
        while( SporeIter != SporeEnd ) {
            (*SporeIter)->getInfo( curr_x, curr_y, curr_energy );
            fprintf(outfile, "%8.2f %8.2f %8.4f\n", curr_x, curr_y, curr_energy );
            SporeIter++;
            numSpores++;
        }

        //print footer
        fprintf(outfile, "#\n#Number of active walkers:   %9i\n", numActive );
        fprintf(outfile,    "#Number of inactive walkers: %9i\n", numSpores );

        fclose(outfile);
    }
    /*************************** end output ****************************/

    std::cout << "Cleaning up...\n";
    //delete walkers
    ActiveIter = ActiveList.begin();
    while( ActiveIter != ActiveEnd )
      delete ( *(ActiveIter++) );

    SporeIter = SporeList.begin();
    while( SporeIter != SporeEnd )
      delete ( *(SporeIter++) );

    //clear lists
    ActiveList.clear();
    SporeList.clear();

    return 0;
}



//goto one char past next '=' and read double parameter
int readParam( FILE* infile, double &parameter )
{
    //look for '=' character in input file
    char c;
    do
        c = fgetc( infile );
    while( !feof( infile ) && c != '=' );
    
    //exit if end of file was reached first
    if( feof( infile ) )
        return 1;
    
    //read the next character
    if( fscanf( infile, "%lf", &parameter ) != 1 ) {
        std::cout << "There was a problem reading the input file.\n";
        return 1;
    }
    return 0;
}

//goto one char past next '=' and read int parameter
int readParam( FILE* infile, int &parameter )
{
    //look for '=' character in input file
    char c;
    do
        c = fgetc( infile );
    while( !feof( infile ) && c != '=' );
    
    //exit if end of file was reached first
    if( feof( infile ) )
        return 1;
    
    //read the next character
    if( fscanf( infile, "%d", &parameter ) != 1 ) {
        std::cout << "There was a problem reading the input file.\n";
        return 1;
    }
    return 0;
}

//goto one past next '#' and read in 3 parameters for inoculation:  # of walkers, x coord, y coord
int readInoc( FILE * infile, int &numWalkers, double &x, double &y )
{
    //look for '#' character in input file
    char c;
    do
        c = fgetc( infile );
    while( !feof( infile ) && c != '#' );
    
    //exit if end of file was reached
    if( feof( infile ) )
        return 1;
    
    //read the 3 parameters
    if( fscanf( infile, "%d %lf %lf", &numWalkers, &x, &y ) != 3 ) {
        std::cout << "There was a problem reading the input file.\n";
        return 1;
    }
    return 0;
}

//general file open function, returns 0 if file was not opened
FILE* openIt( char *filename, char *param )
{
    FILE* openfile;
    if( (openfile = fopen( filename, param )) == NULL ) {
      std::cout <<"Unable to open " << filename << std::endl;
      return NULL;
    }
    return openfile;
}



